/*
  Constant Frequency Sine Wave (DDS) on DAC (SAMD21 XIAO)
  ------------------------------------------------------
  Generates a clean sine wave on DAC A0 at a fixed frequency.

  Uses:
    - Direct Digital Synthesis (DDS)
    - Timer TC5 for precise DAC timing
*/

#include <Arduino.h>
#include <math.h>

// ===== USER PARAMETERS =====
const float toneFreq   = 600.0;   // <<< constant tone frequency (Hz)
const uint16_t amplitude = 2047;  // Max 2047 for centered 12-bit DAC

// ===== INTERNAL CONSTANTS =====
const uint16_t TABLE_SIZE = 256;
const float SAMPLE_RATE = 48000.0;

volatile uint16_t sineTable[TABLE_SIZE];
volatile uint32_t phaseAcc = 0;
volatile uint32_t phaseInc = 0;

// DAC pin
const int dacPin = A0;

// ===== TIMER ISR =====
void TC5_Handler(void) {
  // Clear interrupt flag
  TC5->COUNT16.INTFLAG.bit.MC0 = 1;

  // Advance phase
  phaseAcc += phaseInc;

  // Lookup sine value
  uint16_t index = (phaseAcc >> 24) & 0xFF;
  analogWrite(dacPin, sineTable[index]);
}

// ===== SETUP FUNCTIONS =====
void setupSineTable() {
  for (uint16_t i = 0; i < TABLE_SIZE; i++) {
    float theta = (2.0f * PI * i) / TABLE_SIZE;
    sineTable[i] = amplitude + (uint16_t)(amplitude * sin(theta));
  }
}

void setupTimer() {
  // Enable GCLK for TC5
  GCLK->CLKCTRL.reg = GCLK_CLKCTRL_ID(TC5_GCLK_ID) |
                      GCLK_CLKCTRL_GEN_GCLK0 |
                      GCLK_CLKCTRL_CLKEN;
  while (GCLK->STATUS.bit.SYNCBUSY);

  // Disable timer
  TC5->COUNT16.CTRLA.reg &= ~TC_CTRLA_ENABLE;
  while (TC5->COUNT16.STATUS.bit.SYNCBUSY);

  // Match Frequency mode
  TC5->COUNT16.CTRLA.reg = TC_CTRLA_MODE_COUNT16 |
                           TC_CTRLA_WAVEGEN_MFRQ |
                           TC_CTRLA_PRESCALER_DIV1;

  // Set sample rate
  uint32_t compareValue = (uint32_t)(SystemCoreClock / SAMPLE_RATE);
  TC5->COUNT16.CC[0].reg = compareValue;
  while (TC5->COUNT16.STATUS.bit.SYNCBUSY);

  // Enable interrupt
  NVIC_EnableIRQ(TC5_IRQn);
  TC5->COUNT16.INTENSET.bit.MC0 = 1;

  // Enable timer
  TC5->COUNT16.CTRLA.reg |= TC_CTRLA_ENABLE;
  while (TC5->COUNT16.STATUS.bit.SYNCBUSY);
}

void setup() {
  analogWriteResolution(12);
  pinMode(dacPin, OUTPUT);

  setupSineTable();

  // Compute DDS phase increment ONCE
  phaseInc = (uint32_t)((toneFreq * 4294967296.0) / SAMPLE_RATE);

  setupTimer();
}

// ===== MAIN LOOP =====
void loop() {
  // Nothing needed — tone runs entirely in ISR
}
