#include <WiFi.h>
#include <HTTPClient.h>
#include <ArduinoJson.h>
#include <Adafruit_NeoPixel.h>
#include <ESP_I2S.h>
#include <SPI.h>
#include <Adafruit_GFX.h>
#include <Adafruit_GC9A01A.h>
#include <time.h>
#include <math.h>

// ================= CONFIG =================
const char* ssid = "MIT";
const char* password = "zeky[4rm}H";
const char* serverURL = "http://10.29.253.197:5050/api/transcribe";

#define SAMPLE_RATE     16000
#define FRAME_MS        20
#define FRAME_SAMPLES   (SAMPLE_RATE * FRAME_MS / 1000)
#define SILENCE_MS      900
#define MAX_SECONDS     5
#define BUFFER_SIZE     (SAMPLE_RATE * MAX_SECONDS)

// ================= STATE =================
enum State { IDLE, RECORDING };
State state = IDLE;

// ================= AUDIO =================
int16_t frame[FRAME_SAMPLES];
int16_t audioBuffer[BUFFER_SIZE];
int bufferIndex = 0;

float noise_floor = 0;
uint32_t last_voice_ms = 0;

// ================= I2S =================
I2SClass I2S;

// ================= DISPLAY =================
#define TFT_DC   D3
#define TFT_CS   D5
#define TFT_RST  D6
#define TFT_BL   D2
#define TFT_MOSI D4
#define TFT_SCK  D7
#define TFT_MISO D8

Adafruit_GC9A01A tft(TFT_CS, TFT_DC, TFT_RST);

// ================= LED =================
#define LED_PIN D10
Adafruit_NeoPixel strip(1, LED_PIN, NEO_GRB + NEO_KHZ800);

// ================= WIFI =================
void connectWiFi() {
  WiFi.begin(ssid, password);
  while (WiFi.status() != WL_CONNECTED) delay(500);

  configTime(0, 0, "pool.ntp.org", "time.nist.gov");
}

// ================= RMS =================
float compute_rms(int16_t* buf, int n) {
  double mean = 0, sum = 0;
  for (int i = 0; i < n; i++) mean += buf[i];
  mean /= n;
  for (int i = 0; i < n; i++) {
    double v = buf[i] - mean;
    sum += v * v;
  }
  return sqrt(sum / n);
}

// ================= CENTERED TEXT =================
void drawCenteredText(String text, int size) {
  tft.fillScreen(GC9A01A_BLACK);
  tft.setTextSize(size);
  tft.setTextColor(GC9A01A_WHITE);

  int16_t x1, y1;
  uint16_t w, h;
  tft.getTextBounds(text, 0, 0, &x1, &y1, &w, &h);

  int x = (tft.width() - w) / 2;
  int y = (tft.height() - h) / 2;

  tft.setCursor(x, y);
  tft.print(text);
}

// ================= TIME DISPLAY =================
void showTime() {
  struct tm timeinfo;
  if (!getLocalTime(&timeinfo)) return;

  char buf[32];
  strftime(buf, sizeof(buf), "%H:%M\n%b %d", &timeinfo);
  drawCenteredText(String(buf), 3);
}

// ================= SEND AUDIO =================
bool sendAudio(int16_t* buf, int samples, String &color, String &text) {
  HTTPClient http;
  http.begin(serverURL);
  http.addHeader("Content-Type", "application/octet-stream");

  int res = http.POST((uint8_t*)buf, samples * 2);
  if (res != 200) {
    http.end();
    return false;
  }

  StaticJsonDocument<256> doc;
  deserializeJson(doc, http.getString());
  color = doc["analysis"]["color"].as<String>();
  text  = doc["analysis"]["text"].as<String>();
  http.end();
  return true;
}

// ================= LED FLASH =================
void flashColor(String color) {
  strip.setBrightness(200);

  uint32_t c = 0;
  if (color == "red") c = strip.Color(255,0,0);
  else if (color == "green") c = strip.Color(0,255,0);
  else c = strip.Color(255,255,0);

  strip.setPixelColor(0, c);
  strip.show();
  delay(3000);

  strip.clear();
  strip.show();
}

// ================= SETUP =================
void setup() {
  Serial.begin(115200);

  SPI.begin(TFT_SCK, TFT_MISO, TFT_MOSI, TFT_CS);
  pinMode(TFT_BL, OUTPUT);
  digitalWrite(TFT_BL, HIGH);
  tft.begin();

  strip.begin();
  strip.clear();
  strip.show();

  connectWiFi();

  I2S.setPinsPdmRx(42, 41);
  I2S.begin(I2S_MODE_PDM_RX, SAMPLE_RATE, I2S_DATA_BIT_WIDTH_16BIT, I2S_SLOT_MODE_MONO);
}

// ================= LOOP =================
void loop() {
  for (int i = 0; i < FRAME_SAMPLES; i++) {
    frame[i] = I2S.read();
  }

  float rms = compute_rms(frame, FRAME_SAMPLES);
  noise_floor = 0.99 * noise_floor + 0.01 * rms;
  float threshold = noise_floor * 3.0;

  uint32_t now = millis();

  if (rms > threshold) {
    last_voice_ms = now;

    if (state == IDLE) {
      state = RECORDING;
      bufferIndex = 0;
    }

    if (bufferIndex + FRAME_SAMPLES < BUFFER_SIZE) {
      memcpy(&audioBuffer[bufferIndex], frame, FRAME_SAMPLES * 2);
      bufferIndex += FRAME_SAMPLES;
    }
  }

  if (state == RECORDING && (now - last_voice_ms > SILENCE_MS)) {
    state = IDLE;

    String color, text;
    if (sendAudio(audioBuffer, bufferIndex, color, text)) {
      drawCenteredText(text, 2);
      flashColor(color);
    }

    bufferIndex = 0;
  }

  if (state == IDLE) {
    showTime();
    delay(500);
  }
}

// #include <Adafruit_NeoPixel.h>

// #define LED_PIN 10   // D10
// #define NUM_LEDS 1

// Adafruit_NeoPixel strip(NUM_LEDS, LED_PIN, NEO_GRB + NEO_KHZ800);

// void setup() {
//   Serial.begin(115200);
//   strip.begin();
//   strip.show(); // Initialize all pixels to 'off'
//   strip.setBrightness(200);
// }

// void loop() {
//   // Red
//   strip.setPixelColor(0, strip.Color(255, 0, 0));
//   strip.show();
//   delay(1000);

//   // Green
//   strip.setPixelColor(0, strip.Color(0, 255, 0));
//   strip.show();
//   delay(1000);

//   // Blue
//   strip.setPixelColor(0, strip.Color(0, 0, 255));
//   strip.show();
//   delay(1000);

//   // Off
//   strip.setPixelColor(0, strip.Color(0, 0, 0));
//   strip.show();
//   delay(1000);
// }


// void loop() {
//   // Check Wi-Fi status
//   if (WiFi.status() == WL_CONNECTED) {
//     Serial.println("Wi-Fi connected!");
//     Serial.print("IP address: ");
//     Serial.println(WiFi.localIP());
//   } else {
//     Serial.println("Wi-Fi NOT connected!");
//   }

//   // Read audio frame (optional for testing)
//   for (int i = 0; i < FRAME_SAMPLES; i++) {
//     frame[i] = (int16_t)I2S.read();
//   }

//   // Commented out RMS debug for now
//   // float rms = compute_rms(frame, FRAME_SAMPLES);
//   // if (!speech_active) noise_floor = 0.95f*noise_floor + 0.05f*rms;
//   // float threshold = noise_floor * 3.0f;
//   // Serial.print("RMS: "); Serial.print(rms);
//   // Serial.print("  THR: "); Serial.println(threshold);

//   delay(2000); // print Wi-Fi status every 2 seconds
// }



// #include <SPI.h>
// #include <Adafruit_GFX.h>
// #include <Adafruit_GC9A01A.h>
// #include <ESP_I2S.h>

// I2SClass I2S;

// // ================= DISPLAY PINS =================
// #define TFT_DC   D3
// #define TFT_CS   D5
// #define TFT_RST  D6
// #define TFT_BL   D2

// #define TFT_MOSI D4
// #define TFT_SCK  D7
// #define TFT_MISO D8

// Adafruit_GC9A01A tft(TFT_CS, TFT_DC, TFT_RST);

// // ================= MIC CONFIG ===================
// #define SAMPLE_RATE     16000
// #define FRAME_MS        20
// #define FRAME_SAMPLES   (SAMPLE_RATE * FRAME_MS / 1000)
// #define SILENCE_MS      800

// int16_t frame[FRAME_SAMPLES];

// bool speech_active = false;
// uint32_t last_voice_ms = 0;
// float noise_floor = 0;

// // ================= RMS ==========================
// float compute_rms(int16_t* buf, int n) {
//   double sum = 0;
//   double mean = 0;

//   // Compute mean (DC offset)
//   for (int i = 0; i < n; i++) {
//     mean += buf[i];
//   }
//   mean /= n;

//   // RMS after removing DC
//   for (int i = 0; i < n; i++) {
//     double v = buf[i] - mean;
//     sum += v * v;
//   }

//   return sqrt(sum / n);
// }

// // ================= SETUP ========================
// void setup() {
//   Serial.begin(115200);
//   delay(1000);

//   // -------- SPI / DISPLAY --------
//   SPI.begin(TFT_SCK, TFT_MISO, TFT_MOSI, TFT_CS);

//   pinMode(TFT_BL, OUTPUT);
//   digitalWrite(TFT_BL, HIGH);

//   tft.begin();
//   tft.fillScreen(GC9A01A_BLACK);

//   tft.setTextColor(GC9A01A_WHITE);
//   tft.setTextSize(2);
//   tft.setCursor(20, 20);
//   tft.println("XIAO ESP32S3");
//   tft.println("MIC + TFT TEST");

//   // -------- MIC (PDM I2S) --------
//   I2S.setPinsPdmRx(42, 41); // CLK, DATA

//   if (!I2S.begin(I2S_MODE_PDM_RX,
//                  SAMPLE_RATE,
//                  I2S_DATA_BIT_WIDTH_16BIT,
//                  I2S_SLOT_MODE_MONO)) {
//     tft.setTextColor(GC9A01A_RED);
//     tft.println("I2S INIT FAIL");
//     while (1);
//   }

//   Serial.println("Mic + Display test ready");
// }

// // ================= LOOP =========================
// void loop() {
//   // Read audio frame
//   for (int i = 0; i < FRAME_SAMPLES; i++) {
//     frame[i] = (int16_t)I2S.read();
//   }

//   float rms = compute_rms(frame, FRAME_SAMPLES);

//   // Noise floor estimation
//   if (!speech_active) {
//     noise_floor = 0.95f * noise_floor + 0.05f * rms;
//   }

//   float threshold = noise_floor * 3.0f;
//   uint32_t now = millis();

//   // Speech detection
//   if (rms > threshold) {
//     if (!speech_active) {
//       speech_active = true;
//       Serial.println(">>> SPEECH START");
//     }
//     last_voice_ms = now;
//   } 
//   else if (speech_active && (now - last_voice_ms > SILENCE_MS)) {
//     speech_active = false;
//     Serial.println("<<< SPEECH END");
//   }

//   // -------- DISPLAY UPDATE --------
//   tft.fillRect(0, 80, 240, 120, GC9A01A_BLACK);

//   tft.setCursor(20, 80);
//   tft.setTextColor(GC9A01A_CYAN);
//   tft.setTextSize(2);
//   tft.print("RMS: ");
//   tft.println((int)rms);

//   tft.setCursor(20, 110);
//   if (speech_active) {
//     tft.setTextColor(GC9A01A_GREEN);
//     tft.println("STATUS: SPEECH");
//   } else {
//     tft.setTextColor(GC9A01A_YELLOW);
//     tft.println("STATUS: SILENCE");
//   }

//   // -------- SERIAL DEBUG --------
//   Serial.print("RMS: ");
//   Serial.print(rms);
//   Serial.print("  THR: ");
//   Serial.println(threshold);

//   delay(20);
// }


