# ESP32S3 IMU Wand Path Tracker

A Python server that receives IMU sensor data from an ESP32S3 board via WiFi UDP and tracks the wand tip path in 2D space using Kalman filtering for smooth trajectory visualization.

## Features

- **Kalman-filtered path tracking**: Smooth wand tip trajectory using 2D Kalman filter
- **2D path visualization**: Projects wand movement to xz or yz plane (configurable)
- **Real-time path drawing**: Visual trail showing wand movement history
- **UDP server** for receiving multiple IMU data types:
  - **QUAT**: Quaternion orientation (qw, qx, qy, qz)
  - **ACCEL**: Accelerometer data (x, y, z)
  - **GYRO**: Gyroscope data (x, y, z)
  - **GRAV**: Gravity vector (x, y, z)
- **Console logging** of all received sensor data
- **Clear path button** to reset visualization
- Configurable via environment variables
- Cross-platform (Windows, macOS, Linux)

## Prerequisites

- Python 3.7 or higher
- tkinter (usually bundled with Python)
- ESP32S3 board with WiFi connectivity
- Both devices on the same WiFi network

### Installing tkinter (if needed)

**macOS**: Usually pre-installed with Python

**Linux (Ubuntu/Debian)**:
```bash
sudo apt-get install python3-tk
```

**Linux (Fedora)**:
```bash
sudo dnf install python3-tkinter
```

**Linux (Arch)**:
```bash
sudo pacman -S tk
```

**Windows**: Usually pre-installed with Python

## Installation

1. **Clone or download this repository**

2. **Install Python dependencies**:
   ```bash
   pip install -r requirements.txt
   ```

   Or if using a virtual environment (recommended):
   ```bash
   python3 -m venv venv
   source venv/bin/activate  # On Windows: venv\Scripts\activate
   pip install -r requirements.txt
   ```

3. **Configure settings** (optional):
   ```bash
   cp .env.example .env
   # Edit .env with your preferred settings
   ```

## Configuration

### Server Settings

Edit `.env` file or set environment variables:

- `LISTEN_HOST`: Host to bind UDP server (default: `0.0.0.0` - all interfaces)
- `LISTEN_PORT`: UDP port to listen on (default: `5005`)
- `CANVAS_SIZE`: Size of the canvas window in pixels (default: `800`)
- `REFRESH_HZ`: Display refresh rate in Hz (default: `60.0`)
- `WAND_LENGTH`: Length of wand in arbitrary units (default: `1.0`)
- `MAX_PATH_POINTS`: Maximum number of path points to keep (default: `2000`)
- `PROJECTION_PLANE`: 2D projection plane - `"xz"` or `"yz"` (default: `"xz"`)
- `KALMAN_PROCESS_NOISE`: Kalman filter process noise (default: `0.01`)
- `KALMAN_MEASUREMENT_NOISE`: Kalman filter measurement noise (default: `0.1`)

### ESP32S3 Firmware Settings

Make sure your ESP32S3 firmware has the correct settings:

```cpp
const char* udp_host = "YOUR_PC_IP_ADDRESS";  // Your computer's IP address
const int   udp_port = 5005;                   // Must match LISTEN_PORT
```

**Finding your computer's IP address:**

- **macOS/Linux**: Run `ifconfig` or `ip addr` and look for your WiFi interface IP
- **Windows**: Run `ipconfig` and look for your WiFi adapter IP address
- Make sure both devices are on the same network (e.g., both connected to "MIT" WiFi)

## Usage

1. **Start the server**:
   ```bash
   python3 udp_imu_viewer.py
   ```

   You should see:
   ```
   Listening for UDP packets on 0.0.0.0:5005
   ```

2. **Power on your ESP32S3 board** (with the firmware already uploaded)

3. **The Tkinter window should open** showing:
   - A black canvas with the wand tip path visualization
   - **Cyan line**: Kalman-filtered smooth path (thick, bright)
   - **Gray line**: Raw unfiltered path (thin, dim)
   - **Yellow circle**: Current wand tip position
   - **Colored dots**: Path points with gradient (newer = brighter)
   - All sensor values and path statistics displayed in the top-left corner
   - The path updates in real-time as you move the wand

4. **Check the console** for logged sensor data:
   ```
   [1] QUAT: w=+0.707107, x=+0.000000, y=+0.707107, z=+0.000000
   [2] ACCEL: x=   123, y=   -45, z=   987
   [3] GYRO:  x=    12, y=    34, z=   -56
   [4] GRAV:  x=     0, y=     0, z=  1000
   ```

4. **To stop the server**: Close the Tkinter window or press `Ctrl+C` in the terminal

## Troubleshooting

### No data received / Path not updating

1. **Check network connectivity**:
   - Ensure both devices are on the same WiFi network
   - Verify the ESP32S3 is connected (check Serial Monitor)

2. **Verify IP address**:
   - Make sure `udp_host` in ESP32S3 code matches your computer's IP
   - Check that your computer's firewall allows UDP traffic on port 5005

3. **Check port**:
   - Ensure `LISTEN_PORT` in `.env` matches `udp_port` in ESP32S3 firmware

4. **Test UDP connection**:
   - You can use a UDP testing tool or check the Serial Monitor on ESP32S3 for connection status

### Tkinter window doesn't open

- Install tkinter (see Prerequisites section)
- On some systems, you may need to use `python` instead of `python3`

### Permission denied on port

- On Linux, ports below 1024 require root privileges. Use a port >= 1024 (default is 5005, which should be fine)
- Check if another application is using port 5005

### Firewall blocking UDP packets

**macOS**:
- System Preferences → Security & Privacy → Firewall
- Allow Python when prompted

**Windows**:
- Windows Defender Firewall → Allow an app through firewall
- Add Python to allowed apps

**Linux**:
- Configure your firewall (ufw, firewalld, etc.) to allow UDP on port 5005

## How It Works

The system tracks the wand tip path by:
1. **Receiving quaternion data** from the ESP32S3 via UDP
2. **Calculating wand tip position**:
   - Assumes wand initially points in +Z direction
   - Rotates this direction vector by the quaternion
   - Calculates tip position = base position + (wand_direction × wand_length)
3. **Projecting to 2D plane**: Projects 3D tip position to xz or yz plane (configurable)
4. **Kalman filtering**: Applies a 2D Kalman filter to smooth the path and reduce noise
   - Tracks position and velocity for both axes
   - Uses constant velocity model for prediction
   - Smooths measurements to produce filtered trajectory
5. **Path visualization**: Draws both raw and filtered paths on the canvas

**Note**: IMUs measure orientation, not absolute position. The path is derived from orientation changes, showing the relative movement of the wand tip based on how the device is rotated. The Kalman filter helps smooth out noise and provides a cleaner visualization of the wand's movement pattern.

## Data Format

The ESP32S3 sends UDP packets with different prefixes:

**Quaternion** (orientation):
```
QUAT,0.707107,0.000000,0.707107,0.000000
```

**Accelerometer**:
```
ACCEL,123,-45,987
```

**Gyroscope**:
```
GYRO,12,34,-56
```

**Gravity vector**:
```
GRAV,0,0,1000
```

All values are comma-separated. The server parses each packet type and accumulates the data for visualization.

## Pygame Battle Demo

Alongside the IMU viewer there is a lightweight Pygame mini-game that recreates a turn-based duel inspired by the reference screenshot.

### Features
- Main menu with looping `hp-theme.mp3`
- Battle scene using `background.jpg`, Harry sprite sheet, and Neil boss sprite
- Keyboard abilities mapped to **Q / W / E / R**
- Simple HP bars, damage text, and win/lose flow that returns to the menu
- Automatic music swap to `battle-music.mp3` during fights

### Running the Game

```bash
python3 battle_game.py
```

Controls:
- Menu: `↑` / `↓` + `Enter`
- Battle: `Q`, `W`, `E`, `R` cast abilities, `ESC` returns to menu
- Result screen: `Enter` to go back to menu

> **Sprite sheet note:** `battle_game.py` crops two poses out of `assets/characters/harry.jpg` using the rectangles defined near the top of the script (`HERO_IDLE_RECT` and `HERO_CAST_RECT`). If your copy of the sheet is arranged differently, tweak those rectangles (x, y, width, height) until the intended poses are selected.

## Project Structure

```
imu_server/
├── battle_game.py       # Pygame mock battle with menu + audio
├── udp_imu_viewer.py    # UDP IMU wand path tracker with Kalman filter
├── requirements.txt     # Python dependencies (tkinter/pygame helpers)
├── .env.example         # Example environment configuration
├── .env                 # Your local configuration (create from .env.example)
└── README.md            # This file
```

## License

This project is provided as-is for educational and development purposes.

