#include <WiFi.h>
#include <PubSubClient.h>
#include <HardwareSerial.h>
#include "Adafruit_Thermal.h"

#include <Adafruit_NeoPixel.h>
#include "driver/i2s.h"

// ----------------- PINS -----------------

// Printer on UART1 (D7 = RX, D6 = TX via board macros RX/TX)
#define PRINTER_RX_PIN RX   // D7
#define PRINTER_TX_PIN TX   // D6

// NeoPixels
#define NEOPIXEL_PIN    D3
#define NEOPIXEL_COUNT  8

// I2S (MAX98357)
#define I2S_BCLK_PIN    D1   // matches schematic: D1 = BCLK
#define I2S_LRCLK_PIN   D0   // D0 = LRCLK
#define I2S_DIN_PIN     D2

// ----------------- WIFI / AIO -----------------
// Harcoding the demo WiFi info in
// Censoring my AIO key for this version, since it'll be uploaded online

const char* WIFI_SSID = "CBA";
const char* WIFI_PASS = "Internet";

const char* AIO_USERNAME = "ewolf";
const char* AIO_KEY      = "aio_*****************************";
const char* MQTT_HOST    = "io.adafruit.com";
const uint16_t MQTT_PORT = 1883;

String FEED        = "oled-text";
String TOPIC_RAW   = String(AIO_USERNAME) + "/feeds/" + FEED;
String TOPIC_JSON  = TOPIC_RAW + "/json";

// ----------------- OBJECTS -----------------

WiFiClient wifi;
PubSubClient mqtt(wifi);

// Printer (hardware serial only)
HardwareSerial mySerial(1);
Adafruit_Thermal printer(&mySerial);

// NeoPixels
Adafruit_NeoPixel strip(NEOPIXEL_COUNT, NEOPIXEL_PIN, NEO_GRB + NEO_KHZ800);

// ----------------- NEOPIXEL + AUDIO HELPERS -----------------

void neopixelChase() {
  strip.clear();
  strip.setBrightness(80);

  // simple one-pixel chase across the 8 LEDs
  for (int i = 0; i < NEOPIXEL_COUNT; i++) {
    strip.clear();
    strip.setPixelColor(i, strip.Color(0, 150, 255));  // cyan
    strip.show();
    delay(80);
  }

  // flash all on briefly, then off
  for (int i = 0; i < NEOPIXEL_COUNT; i++) {
    strip.setPixelColor(i, strip.Color(0, 150, 255));
  }
  strip.show();
  delay(200);
  strip.clear();
  strip.show();
}

void i2sInitOnce() {
  static bool inited = false;
  if (inited) return;

  i2s_config_t i2s_config = {
    .mode = (i2s_mode_t)(I2S_MODE_MASTER | I2S_MODE_TX),
    .sample_rate = 44100,
    .bits_per_sample = I2S_BITS_PER_SAMPLE_16BIT,
    .channel_format = I2S_CHANNEL_FMT_ONLY_LEFT,
    .communication_format = (i2s_comm_format_t)I2S_COMM_FORMAT_STAND_I2S,
    .intr_alloc_flags = 0,
    .dma_buf_count = 4,
    .dma_buf_len = 256,
    .use_apll = false,
    .tx_desc_auto_clear = true,
    .fixed_mclk = 0
  };

  i2s_pin_config_t pins = {
    .bck_io_num   = I2S_BCLK_PIN,
    .ws_io_num    = I2S_LRCLK_PIN,
    .data_out_num = I2S_DIN_PIN,
    .data_in_num  = I2S_PIN_NO_CHANGE
  };

  i2s_driver_install(I2S_NUM_0, &i2s_config, 0, NULL);
  i2s_set_pin(I2S_NUM_0, &pins);
  i2s_set_clk(I2S_NUM_0, 44100, I2S_BITS_PER_SAMPLE_16BIT, I2S_CHANNEL_MONO);

  inited = true;
}

void playBeep() {
  i2sInitOnce();

  const int sampleRate = 44100;
  const float freq     = 440.0f;
  const float duration = 0.15f;      // seconds
  int totalSamples     = (int)(sampleRate * duration);

  for (int i = 0; i < totalSamples; i++) {
    float t      = (float)i / sampleRate;
    float s      = sinf(2.0f * PI * freq * t);   // -1..1
    int16_t smpl = (int16_t)(s * 30000);         // scale to 16-bit

    size_t written;
    i2s_write(I2S_NUM_0, &smpl, sizeof(smpl), &written, portMAX_DELAY);
  }
}

// ----------------- PRINTER / MQTT HELPERS -----------------

void printMessage(const String& msg) {
  printer.println(msg);
  printer.feed(3);
}

void onMsg(char* topic, byte* payload, unsigned int len) {
  String m;
  m.reserve(len);
  for (unsigned i = 0; i < len; i++) {
    m += (char)payload[i];
  }

  // Extract "value" if JSON payload from Adafruit IO
  String t(topic);
  if (t.endsWith("/json")) {
    int i = m.indexOf("\"value\"");
    if (i >= 0) {
      int c  = m.indexOf(':', i);
      int q1 = m.indexOf('"', c + 1);
      int q2 = m.indexOf('"', q1 + 1);
      if (q1 >= 0 && q2 > q1) {
        m = m.substring(q1 + 1, q2);
      }
    }
  }

  Serial.printf("MQTT IN [%s] (%u bytes): %s\n", topic, len, m.c_str());

  // New behavior: lights + sound + print
  neopixelChase();
  playBeep();
  printMessage(m);
}

void connectWiFi() {
  Serial.printf("Connecting Wi-Fi to \"%s\"...\n", WIFI_SSID);
  WiFi.mode(WIFI_STA);
  WiFi.begin(WIFI_SSID, WIFI_PASS);
  unsigned long t0 = millis();
  while (WiFi.status() != WL_CONNECTED && millis() - t0 < 20000) {
    delay(250);
    Serial.print(".");
  }
  Serial.println();
  if (WiFi.status() == WL_CONNECTED) {
    Serial.print("Wi-Fi OK, IP: ");
    Serial.println(WiFi.localIP());
  } else {
    Serial.println("Wi-Fi FAILED");
  }
}

bool connectMQTT() {
  mqtt.setServer(MQTT_HOST, MQTT_PORT);
  mqtt.setCallback(onMsg);

  String clientId = "xiao-c3-" + String((uint32_t)ESP.getEfuseMac(), HEX);
  Serial.printf("Connecting MQTT to %s:%u as %s\n",
                MQTT_HOST, MQTT_PORT, clientId.c_str());

  bool ok = mqtt.connect(clientId.c_str(), AIO_USERNAME, AIO_KEY);
  if (!ok) {
    Serial.printf("MQTT connect FAILED, state=%d\n", mqtt.state());
    return false;
  }

  Serial.println("MQTT connected");
  Serial.println("Subscribing to:");
  Serial.println("  " + TOPIC_RAW);
  Serial.println("  " + TOPIC_JSON);
  mqtt.subscribe(TOPIC_RAW.c_str());
  mqtt.subscribe(TOPIC_JSON.c_str());
  return true;
}

unsigned long lastTry = 0;

// ----------------- SETUP / LOOP -----------------

void setup() {
  Serial.begin(115200);
  delay(200);

  // NeoPixels
  strip.begin();
  strip.show(); // all off

  // Printer UART (HardwareSerial on UART1)
  mySerial.begin(19200, SERIAL_8N1, PRINTER_RX_PIN, PRINTER_TX_PIN);
  printer.begin();
  printer.println("Booting...");
  printer.feed(2);

  connectWiFi();
  connectMQTT();

  Serial.println("Ready. Publish to your Adafruit IO feed to print + lights + beep.");
}

void loop() {
  if (WiFi.status() != WL_CONNECTED && millis() - lastTry > 3000) {
    lastTry = millis();
    connectWiFi();
  }

  if (!mqtt.connected() && millis() - lastTry > 3000) {
    lastTry = millis();
    connectMQTT();
  }

  mqtt.loop();
}
