#include <Arduino.h>
#include <ESP_I2S.h>
#include <math.h>

// -------- Global I2S instance --------
I2SClass I2S;

// -------- Button pin --------
const int BUTTON_PIN = 4;   // tactile button on GPIO 4 (to GND)

// -------- Mic pins (XIAO ESP32S3 Sense) --------
const int MIC_CLK_PIN  = 42;   // PDM clock
const int MIC_DATA_PIN = 41;   // PDM data

// -------- Speaker pins (Adafruit I2S BFF) --------
const int SPK_BCLK_PIN  = 2;   // BCLK
const int SPK_LRCLK_PIN = 3;   // LRCLK / WS
const int SPK_DATA_PIN  = 1;   // DIN

// -------- Audio parameters --------
const int SAMPLE_RATE      = 16000;         // use same rate for mic & speaker
const int MAX_RECORD_MS    = 2000;         // max 2 seconds
const int MAX_SAMPLES      = (SAMPLE_RATE * MAX_RECORD_MS) / 1000;

// buffer to store recorded audio
int16_t recordBuffer[MAX_SAMPLES];
size_t  recordedSamples = 0;

enum I2SMode {
  MODE_MIC,
  MODE_SPEAKER
};

I2SMode currentMode = MODE_MIC;
bool isRecording = false;

// ---------- MIC SETUP (PDM RX) ----------
bool startMic() {
  Serial.println("Configuring I2S for PDM mic...");

  I2S.end();  // safely deinit any previous mode

  I2S.setPinsPdmRx(MIC_CLK_PIN, MIC_DATA_PIN);

  bool ok = I2S.begin(
      I2S_MODE_PDM_RX,
      SAMPLE_RATE,                 // 16 kHz
      I2S_DATA_BIT_WIDTH_16BIT,
      I2S_SLOT_MODE_MONO
  );

  if (!ok) {
    Serial.println("Failed to initialize I2S mic!");
    return false;
  }

  Serial.println("I2S mic started.");
  currentMode = MODE_MIC;
  return true;
}

// ---------- SPEAKER SETUP (STD TX) ----------
bool startSpeaker() {
  Serial.println("Configuring I2S for speaker...");

  I2S.end();  // stop mic mode

  // bclk, ws, dout, din=-1
  I2S.setPins(SPK_BCLK_PIN, SPK_LRCLK_PIN, SPK_DATA_PIN, -1);

  bool ok = I2S.begin(
      I2S_MODE_STD,
      SAMPLE_RATE,                  // must match recording rate
      I2S_DATA_BIT_WIDTH_16BIT,
      I2S_SLOT_MODE_MONO
  );

  if (!ok) {
    Serial.println("Failed to initialize I2S speaker!");
    return false;
  }

  currentMode = MODE_SPEAKER;
  return true;
}

// ---------- PLAY BACK RECORDED BUFFER ----------
void playRecordedAudio() {
  if (recordedSamples == 0) {
    Serial.println("Nothing recorded, skipping playback.");
    return;
  }

  if (!startSpeaker()) return;

  Serial.print("Playing back ");
  Serial.print(recordedSamples);
  Serial.println(" samples...");

  for (size_t i = 0; i < recordedSamples; i++) {
    int32_t s = recordBuffer[i];

    // boost volume a bit (x2), clipping-safe
    s *= 8;
    if (s > 32767)  s = 32767;
    if (s < -32768) s = -32768;
    int16_t sample = (int16_t)s;

    I2S.write((uint8_t*)&sample, sizeof(sample));
  }

  Serial.println("Playback finished.");
  // go back to mic mode for next recording
  startMic();
}

void setup() {
  Serial.begin(115200);
  while (!Serial) {
    delay(10);
  }

  Serial.println("Button-controlled mic + speaker test (ESP_I2S)");
  Serial.println("Hold button on GPIO 4 to record, release to play.");

  pinMode(BUTTON_PIN, INPUT_PULLUP);   // button to GND

  startMic();
}

void loop() {
  // ----- Read button state (active LOW) -----
  int buttonState = digitalRead(BUTTON_PIN);

  // Button pressed (transition HIGH -> LOW)
  static int lastButtonState = HIGH;
  if (buttonState != lastButtonState) {
    delay(10); // simple debounce
    buttonState = digitalRead(BUTTON_PIN);
  }

  // Start recording when button is pressed
  if (buttonState == LOW && !isRecording) {
    Serial.println("Button pressed -> start recording");
    if (currentMode != MODE_MIC) {
      startMic();
    }
    isRecording = true;
    recordedSamples = 0;
  }

  // Stop recording & play when button released
  if (buttonState == HIGH && isRecording) {
    Serial.println("Button released -> stop recording & play");
    isRecording = false;
    playRecordedAudio();
  }

  lastButtonState = buttonState;

  // ----- If recording, grab samples from mic -----
  if (isRecording && currentMode == MODE_MIC && recordedSamples < MAX_SAMPLES) {
    int sample = I2S.read();  // blocking read of one sample

    // You can skip filtering; here we record everything
    recordBuffer[recordedSamples++] = (int16_t)(sample*2);
  }

  // (Optional) small delay if you want to reduce CPU usage
  // delay(1);
}