#include <Arduino.h>
#include "driver/i2s.h"
#include <math.h>

// ------- I2S pins for XIAO ESP32-S3 + Adafruit I2S BFF -------
#define I2S_PORT   I2S_NUM_0
#define I2S_BCLK   2   // BCLK
#define I2S_LRCLK  3   // LRCLK / WS
#define I2S_DATA   1   // DIN

const int SAMPLE_RATE = 44100;

// --------- I2S SETUP ----------
void setupI2S() {
  i2s_config_t i2s_config = {
    .mode = (i2s_mode_t)(I2S_MODE_MASTER | I2S_MODE_TX),
    .sample_rate = SAMPLE_RATE,
    .bits_per_sample = I2S_BITS_PER_SAMPLE_16BIT,
    .channel_format = I2S_CHANNEL_FMT_ONLY_RIGHT,      // mono
    .communication_format = I2S_COMM_FORMAT_I2S,
    .intr_alloc_flags = ESP_INTR_FLAG_LEVEL1,
    .dma_buf_count = 8,
    .dma_buf_len = 64,
    .use_apll = false,
    .tx_desc_auto_clear = true,
    .fixed_mclk = 0
  };

  i2s_pin_config_t pin_config = {
    .bck_io_num = I2S_BCLK,
    .ws_io_num = I2S_LRCLK,
    .data_out_num = I2S_DATA,
    .data_in_num = I2S_PIN_NO_CHANGE
  };

  i2s_driver_install(I2S_PORT, &i2s_config, 0, NULL);
  i2s_set_pin(I2S_PORT, &pin_config);
  i2s_zero_dma_buffer(I2S_PORT);
}

// --------- PLAY A TONE (ONE "SOUND") ----------
void playTone(float freq, uint32_t durationMs, int16_t amplitude = 8000) {
  uint32_t totalSamples = (uint32_t)((SAMPLE_RATE * (uint64_t)durationMs) / 1000);
  size_t bytes_written;

  float phase = 0.0f;
  float phaseIncrement = 2.0f * PI * freq / SAMPLE_RATE;

  const int BUFFER_SAMPLES = 256;
  
  int16_t buffer[BUFFER_SAMPLES];

  uint32_t samplesGenerated = 0;

  while (samplesGenerated < totalSamples) {
    int samplesThisLoop = min((uint32_t)BUFFER_SAMPLES, totalSamples - samplesGenerated);

    for (int i = 0; i < samplesThisLoop; i++) {
      buffer[i] = (int16_t)(amplitude * sinf(phase));
      phase += phaseIncrement;
      if (phase > 2.0f * PI) phase -= 2.0f * PI;
    }

    i2s_write(I2S_PORT, buffer, samplesThisLoop * sizeof(int16_t),
              &bytes_written, portMAX_DELAY);

    samplesGenerated += samplesThisLoop;
  }
}

// --------- ARDUINO SETUP & LOOP ----------
void setup() {
  Serial.begin(115200);
  delay(1000);
  Serial.println("I2S multi-sound test ready.");
  Serial.println("Type 1, 2, or 3 in Serial Monitor and press Enter.");

  setupI2S();
}

void loop() {
  if (Serial.available()) {
    char c = Serial.read();

    if (c == '1') {
      Serial.println("Playing sound 1 (440 Hz)…");
      playTone(440.0f, 500);          // A4 beep, 0.5s
    } else if (c == '2') {
      Serial.println("Playing sound 2 (660 Hz)…");
      playTone(660.0f, 500);          // higher beep
    } else if (c == '3') {
      Serial.println("Playing sound 3 (short triple beep)…");
      for (int i = 0; i < 3; i++) {
        playTone(523.25f, 150);       // C5
        delay(80);
      }
    }
  }
}