/**
 * XIAO ESP32S3 + RC522 + PDM Mic + I2S Speaker + UDP
 *
 * Flow:
 *   1) Wait for RFID scan (UID)
 *   2) Beep
 *   3) Connect WiFi
 *   4) Send UID to server once (UDP "UID|<uid>")
 *   5) Button hold = mic stream to server (UDP)
 *   6) Speaker downlink plays PCM when received ("PLAY" + PCM + "STOP")
 *
 * Minimal-change stability fixes:
 *   - Do NOT read RFID during playback
 *   - NO gain on speaker path
 *   - yield() during long loops
 */

#include <Arduino.h>
#include <WiFi.h>
#include <WiFiUdp.h>
#include <ESP_I2S.h>
#include <SPI.h>
#include <MFRC522.h>

// ========================= CONFIG =========================

// ---- WiFi ----
const char* WIFI_SSID = "YOUR WIFI";
const char* WIFI_PASS = "YOUR PASSWORD";

// ---- Button ----
const int BUTTON_PIN = 4;
const int DEBOUNCE_THRESHOLD = 5;

// ---- Mic (PDM) ----
I2SClass I2S;
const int MIC_CLK_PIN  = 42;
const int MIC_DATA_PIN = 41;

// ---- Speaker (I2S TX) ----
const int SPK_BCLK_PIN  = 2;   // BCLK
const int SPK_LRCLK_PIN = 3;   // LRCLK / WS
const int SPK_DATA_PIN  = 1;   // DIN

// ---- RFID SPI pins (your working mapping) ----
const int RFID_SS_PIN   = 5;
const int RFID_RST_PIN  = 43;
const int RFID_SCK_PIN  = 7;
const int RFID_MOSI_PIN = 9;
const int RFID_MISO_PIN = 8;

MFRC522 mfrc522(RFID_SS_PIN, RFID_RST_PIN);
String userId = "UNKNOWN";
bool uidSentToServer = false;

// ---- Audio parameters ----
const int SAMPLE_RATE   = 16000;
const int CHUNK_SAMPLES = 256;
const int CHUNK_BYTES   = CHUNK_SAMPLES * 2; // mono int16

// ---- UDP ----
WiFiUDP udpOut;
WiFiUDP udpIn;
IPAddress udpAddress(YOUR ADDRESS);   // server IP
const int UDP_UPLINK_PORT   = 8000;     // server port for uplink
const int UDP_DOWNLINK_PORT = 9999;     // ESP listens for PCM downlink here

// UID header size used in your server code (keep consistent)
// NOTE: You previously used 16 in server. Keep 16 here too.
const int UID_HEADER_SIZE = 16;

// ========================= STATE MACHINE =========================

enum State {
  STATE_WAIT_RFID,
  STATE_WIFI_CONNECT,
  STATE_READY,
  STATE_RECORDING
};

State state = STATE_WAIT_RFID;

// Button debounce
int  buttonCounter   = 0;
bool buttonState     = HIGH;
bool lastButtonState = HIGH;

// Audio mode flags
bool i2sInMicMode = true;
bool isPlaybackMode = false;

// ========================= I2S HELPERS =========================

bool startMic() {
  Serial.println("[I2S] Init PDM mic mode...");
  I2S.end();

  I2S.setPinsPdmRx(MIC_CLK_PIN, MIC_DATA_PIN);

  if (!I2S.begin(I2S_MODE_PDM_RX,
                 SAMPLE_RATE,
                 I2S_DATA_BIT_WIDTH_16BIT,
                 I2S_SLOT_MODE_MONO)) {
    Serial.println("[I2S] Failed to init mic!");
    return false;
  }

  i2sInMicMode = true;
  Serial.println("[I2S] Mic mode ready.");
  return true;
}

bool startSpeaker() {
  Serial.println("[I2S] Speaker TX mode...");
  I2S.end();

  // bclk, ws, dout, din=-1
  I2S.setPins(SPK_BCLK_PIN, SPK_LRCLK_PIN, SPK_DATA_PIN, -1);

  // NOTE: Your downlink handler expands mono->stereo, so use STEREO here.
  if (!I2S.begin(I2S_MODE_STD,
                 SAMPLE_RATE,
                 I2S_DATA_BIT_WIDTH_16BIT,
                 I2S_SLOT_MODE_STEREO)) {
    Serial.println("[I2S] Speaker init failed");
    return false;
  }

  i2sInMicMode = false;
  Serial.println("[I2S] Speaker mode ready.");
  return true;
}

// ========================= WIFI HELPERS =========================

bool connectWiFiBlocking() {
  Serial.println("[WiFi] Connecting...");
  WiFi.mode(WIFI_STA);
  WiFi.begin(WIFI_SSID, WIFI_PASS);

  int attempts = 0;
  while (WiFi.status() != WL_CONNECTED && attempts < 40) {
    delay(500);
    Serial.print(".");
    attempts++;
    yield();
  }
  Serial.println();

  if (WiFi.status() == WL_CONNECTED) {
    Serial.println("[WiFi] Connected!");
    Serial.print("[WiFi] IP: ");
    Serial.println(WiFi.localIP());
    return true;
  }

  Serial.println("[WiFi] Failed to connect.");
  return false;
}

// ========================= RFID HELPERS =========================

bool readRfidUidOnce(String& outUid) {
  if (!mfrc522.PICC_IsNewCardPresent()) return false;
  if (!mfrc522.PICC_ReadCardSerial())   return false;

  outUid = "";
  for (byte i = 0; i < mfrc522.uid.size; i++) {
    if (mfrc522.uid.uidByte[i] < 0x10) outUid += "0";
    outUid += String(mfrc522.uid.uidByte[i], HEX);
  }
  outUid.toUpperCase();

  mfrc522.PICC_HaltA();
  mfrc522.PCD_StopCrypto1();
  return true;
}

// ========================= UID REGISTRATION =========================

bool sendUidToServerOnce() {
  if (uidSentToServer) return true;
  if (WiFi.status() != WL_CONNECTED) return false;

  String msg = "UID|" + userId;
  if (udpOut.beginPacket(udpAddress, UDP_UPLINK_PORT) != 1) {
    Serial.println("[UID] beginPacket failed");
    return false;
  }
  udpOut.write((const uint8_t*)msg.c_str(), msg.length());
  if (udpOut.endPacket() != 1) {
    Serial.println("[UID] endPacket failed");
    return false;
  }

  uidSentToServer = true;
  Serial.print("[UID] Sent to server: ");
  Serial.println(msg);
  return true;
}

// ========================= SPEAKER BEEP (on RFID) =========================

void beepOnSpeaker() {
  // Use speaker mode briefly to play a short beep tone
  // This is intentionally small + yields so it doesn't break RFID.

  if (!startSpeaker()) return;
  isPlaybackMode = true;

  const float freq = 880.0f;     // A5-ish
  const float durS = 0.12f;      // 120ms
  const int n = (int)(SAMPLE_RATE * durS);

  for (int i = 0; i < n; i++) {
    float t = (float)i / (float)SAMPLE_RATE;
    float s = sinf(2.0f * PI * freq * t);
    int16_t v = (int16_t)(s * 12000); // modest volume
    // stereo frame L,R
    int16_t frame[2] = { v, v };
    I2S.write((uint8_t*)frame, sizeof(frame));

    if ((i & 0x3F) == 0) yield();
  }

  // Switch back to mic
  isPlaybackMode = false;
  startMic();
}

// ========================= UDP DOWNLINK (SPEAKER) =========================

void handleDownlink() {
  if (WiFi.status() != WL_CONNECTED) return;

  int packetSize = udpIn.parsePacket();
  if (packetSize <= 0) return;

  static uint8_t buf[CHUNK_BYTES + 16];
  int len = udpIn.read(buf, sizeof(buf));
  if (len <= 0) return;

  // Control packets
  if (len == 4 && memcmp(buf, "PLAY", 4) == 0) {
    Serial.println("[DOWNLINK] Control: PLAY");
    isPlaybackMode = startSpeaker();
    return;
  }
  if (len == 4 && memcmp(buf, "STOP", 4) == 0) {
    Serial.println("[DOWNLINK] Control: STOP");
    isPlaybackMode = false;
    startMic();
    return;
  }

  if (!isPlaybackMode) return;

  // PCM mono -> expand to stereo and write
  int numSamples = len / 2;
  if (numSamples <= 0) return;
  if (numSamples > CHUNK_SAMPLES) numSamples = CHUNK_SAMPLES;

  static int16_t monoBuf[CHUNK_SAMPLES];
  memcpy(monoBuf, buf, numSamples * 2);

  static int16_t stereoBuf[CHUNK_SAMPLES * 2];
  for (int i = 0; i < numSamples; i++) {
    int16_t s = monoBuf[i];
    stereoBuf[2 * i]     = s;
    stereoBuf[2 * i + 1] = s;

    if ((i & 0x3F) == 0) yield(); // keep system responsive
  }

  // Write whole chunk at once (no gain here!)
  I2S.write((uint8_t*)stereoBuf, numSamples * 4);
}

// ========================= SETUP =========================

void setup() {
  Serial.begin(115200);
  while (!Serial) delay(10);

  Serial.println("\n=== XIAO ESP32S3 RFID -> WiFi -> UID -> Mic/Speaker UDP ===");

  pinMode(BUTTON_PIN, INPUT_PULLUP);

  // IMPORTANT: use custom SPI pins for XIAO ESP32S3 (your working mapping)
  SPI.begin(RFID_SCK_PIN, RFID_MISO_PIN, RFID_MOSI_PIN, RFID_SS_PIN);
  mfrc522.PCD_Init();
  Serial.println("[RFID] Ready. Tap a tag to start WiFi...");

  state = STATE_WAIT_RFID;
}

// ========================= LOOP =========================

void loop() {
  // Downlink always (only does anything after WiFi + udpIn.begin)
  handleDownlink();

  switch (state) {

    case STATE_WAIT_RFID: {
      // Only read RFID when NOT playing and mic mode is active
      if (state == STATE_WAIT_RFID && !isPlaybackMode && i2sInMicMode) {
        String uid;
        if (readRfidUidOnce(uid)) {
          userId = uid;
          Serial.print("[RFID] UID = ");
          Serial.println(userId);

          // beep to confirm scan
          beepOnSpeaker();

          state = STATE_WIFI_CONNECT;
        }
      }
      break;
    }

    case STATE_WIFI_CONNECT: {
      if (connectWiFiBlocking()) {
        // Start mic + UDP sockets
        startMic();
        udpOut.begin(5005);                 // any local port
        udpIn.begin(UDP_DOWNLINK_PORT);     // listen for TTS downlink

        // Send UID once
        sendUidToServerOnce();

        Serial.print("[UDP] Uplink to ");
        Serial.print(udpAddress);
        Serial.print(":");
        Serial.println(UDP_UPLINK_PORT);

        Serial.print("[UDP] Downlink listening on ");
        Serial.println(UDP_DOWNLINK_PORT);

        state = STATE_READY;
      } else {
        delay(3000);
      }
      break;
    }

    case STATE_READY: {
      int raw = digitalRead(BUTTON_PIN);

      if (raw == LOW) {
        buttonCounter++;
        if (buttonCounter >= DEBOUNCE_THRESHOLD) {
          buttonState = LOW;
          buttonCounter = DEBOUNCE_THRESHOLD;
        }
      } else {
        buttonCounter--;
        if (buttonCounter <= -DEBOUNCE_THRESHOLD) {
          buttonState = HIGH;
          buttonCounter = -DEBOUNCE_THRESHOLD;
        }
      }

      if (buttonState != lastButtonState) {
        if (buttonState == LOW) {
          Serial.println("Speaking... (start UDP stream)");
          state = STATE_RECORDING;
        }
        lastButtonState = buttonState;
      }
      break;
    }

    case STATE_RECORDING: {
      int raw = digitalRead(BUTTON_PIN);

      if (raw == LOW) {
        buttonCounter++;
        if (buttonCounter >= DEBOUNCE_THRESHOLD) {
          buttonState = LOW;
          buttonCounter = DEBOUNCE_THRESHOLD;
        }
      } else {
        buttonCounter--;
        if (buttonCounter <= -DEBOUNCE_THRESHOLD) {
          buttonState = HIGH;
          buttonCounter = -DEBOUNCE_THRESHOLD;
        }
      }

      // Only stream when button held, WiFi ok, mic mode, and not currently playing
      if (buttonState == LOW && WiFi.status() == WL_CONNECTED && i2sInMicMode && !isPlaybackMode) {
        int16_t chunkBuf[CHUNK_SAMPLES];

        for (int i = 0; i < CHUNK_SAMPLES; i++) {
          int32_t s = (int32_t)I2S.read();

          // ✅ mic gain (adjust here)
          // start with 8, try 12 if still quiet
          s *= 10;

          // soft clip
          if (s > 30000)  s = 30000 + (s - 30000) / 6;
          if (s < -30000) s = -30000 + (s + 30000) / 6;

          if (s > 32767)  s = 32767;
          if (s < -32768) s = -32768;

          chunkBuf[i] = (int16_t)s;

          if ((i & 0x3F) == 0) yield();
        }

        // Packet = 16-byte UID header + PCM16 payload
        const int PACKET_SIZE = UID_HEADER_SIZE + (CHUNK_SAMPLES * 2);
        uint8_t packet[PACKET_SIZE];

        memset(packet, 0, UID_HEADER_SIZE);
        size_t copyLen = min(userId.length(), (size_t)UID_HEADER_SIZE);
        memcpy(packet, userId.c_str(), copyLen);
        memcpy(packet + UID_HEADER_SIZE, (uint8_t*)chunkBuf, CHUNK_SAMPLES * 2);

        if (udpOut.beginPacket(udpAddress, UDP_UPLINK_PORT) == 1) {
          udpOut.write(packet, PACKET_SIZE);
          udpOut.endPacket();
        }
      }

      if (buttonState == HIGH && lastButtonState == LOW) {
        Serial.println("[UDP] Sent (stop UDP stream)");
        state = STATE_READY;
      }

      lastButtonState = buttonState;
      break;
    }
  }

  yield();
}