#include <WiFi.h>
#include <WebServer.h>
#include "HX711.h"

// ---------------- WIFI ----------------
const char* ssid     = "EDS_Shop_24";
const char* password = "eds12345";
//const char* ssid     = "MIT";
//const char* password = "kwN4@9{UwQ";

// ---------------- HX711 ----------------
#define HX711_DT   A1    // D1
#define HX711_SCK  A0    // D0

HX711 scale;
float calibration_factor = 2280.0f;
float gripStrength = 0;

// ---------------- WEB SERVER ----------------
WebServer server(80);

// ---------------- Age Estimation ----------------
float ageFromGripMale(float g) {
    float a = 52.0;
    float b = 0.0089;
    if (g > a) g = a;
    if (g < 5) g = 5;
    float term = (a - g) / b;
    float root = sqrt(term);
    return 35 + root;
}

float ageFromGripFemale(float g) {
    float a = 32.0;
    float b = 0.0059;
    if (g > a) g = a;
    if (g < 4) g = 4;
    float term = (a - g) / b;
    float root = sqrt(term);
    return 35 + root;
}

// GUI HTML + Chart.js
const char GUI_PAGE[] PROGMEM =
"<!DOCTYPE html>"
"<html><head><title>Grip Strength</title>"
"<script src=\"https://cdn.jsdelivr.net/npm/chart.js\"></script>"
"<style>"
"body{background:#111;color:#fff;font-family:Arial;text-align:center;}"
"h1{color:#9cf;margin-top:25px;}"
".box{width:300px;margin:20px auto;background:#223;padding:20px;"
"border-radius:12px;border:1px solid #4af;}"
".value{font-size:48px;color:#3f9;margin-top:10px;}"
".label{font-size:20px;color:#9cf;}"
".row{display:flex;justify-content:center;gap:20px;margin-top:25px;}"
".chartBox{width:420px;height:260px;background:#222;padding:10px;border-radius:12px;}"
".btn{padding:10px 20px;background:#444;color:#fff;border:1px solid #777;"
"border-radius:8px;font-size:18px;cursor:pointer;margin:10px;}"
".btnOn{background:#3af;}"
"</style>"
"</head><body>"

"<h1>Grip Strength Monitor</h1>"

"<div class='box'>"
"<div class='label'>Sex</div>"
"<button class='btn btnOn' id='maleBtn' onclick='setSex(\"male\")'>Male</button>"
"<button class='btn' id='femaleBtn' onclick='setSex(\"female\")'>Female</button>"
"</div>"

"<div class='box'>"
"<div class='label'>Estimated Age</div>"
"<div id='ageVal' class='value'>--</div>"
"</div>"

"<div class='box'>"
"<div class='label'>Grip Strength (kg)</div>"
"<div id='gripVal' class='value'>--</div>"
"</div>"

"<div class='row'>"
"<div class='chartBox'><canvas id='gripChart'></canvas></div>"
"</div>"

"<script>"
"let gripData=[]; const maxPoints=120;"
"let sex='male';"

"function setSex(s){"
" sex=s;"
" document.getElementById('maleBtn').classList.remove('btnOn');"
" document.getElementById('femaleBtn').classList.remove('btnOn');"
" if(s==='male') document.getElementById('maleBtn').classList.add('btnOn');"
" else document.getElementById('femaleBtn').classList.add('btnOn');"
"}"

"const gripChart=new Chart(document.getElementById('gripChart'),{"
" type:'line',"
" data:{labels:[],datasets:[{label:'Grip (kg)',data:[],borderColor:'#3af'}]},"
" options:{scales:{x:{display:false},y:{min:0,max:80}}}"
"});"

"function update(){"
" fetch('/metrics').then(r=>r.json()).then(data=>{"
"  document.getElementById('gripVal').innerHTML = data.grip.toFixed(1);"
"  document.getElementById('ageVal').innerHTML  = data.age.toFixed(0);"

"  gripData.push(data.grip); if(gripData.length>maxPoints) gripData.shift();"
"  gripChart.data.labels = gripData.map((_,i)=>i);"
"  gripChart.data.datasets[0].data = gripData;"
"  gripChart.update();"
" });"
"}"

"setInterval(update,200);"
"</script>"
"</body></html>";

void handleGUI() {
    server.sendHeader("Cache-Control", "no-cache");
    server.send_P(200, "text/html; charset=utf-8", GUI_PAGE);
}

void handleMetrics() {
    String sex = server.hasArg("sex") ? server.arg("sex") : "male";

    float age = (sex == "female") ? ageFromGripFemale(gripStrength)
                                  : ageFromGripMale(gripStrength);

    String json = "{";
    json += "\"grip\":" + String(gripStrength,1) + ",";
    json += "\"age\":" + String(age,1);
    json += "}";
    server.send(200, "application/json", json);
}

void setup() {
    Serial.begin(115200);
    delay(300);

    // HX711
    scale.begin(HX711_DT, HX711_SCK);
    scale.set_scale(calibration_factor);
    scale.tare();

    // WiFi
    WiFi.begin(ssid, password);
    Serial.print("Connecting to WiFi");
    while(WiFi.status() != WL_CONNECTED){
        delay(300); Serial.print(".");
    }
    Serial.print("\nIP: "); Serial.println(WiFi.localIP());

    // Webserver
    server.on("/", handleGUI);
    server.on("/metrics", handleMetrics);
    server.begin();
}

unsigned long lastRead=0;

void loop() {
    server.handleClient();

    unsigned long now = millis();
    if(now - lastRead > 200){
        lastRead = now;
        if(scale.is_ready()){
            gripStrength = scale.get_units(3);  // smoothed
            if(gripStrength < 0) gripStrength = 0;
        }
    }
}
