/**
 * camera_config.h - Camera initialization for XIAO ESP32-S3 Sense
 */

#ifndef CAMERA_CONFIG_H
#define CAMERA_CONFIG_H

#include <esp_camera.h>
#include <Arduino.h>
#include "vision/image.h"

// XIAO ESP32-S3 Sense camera pins
#define CAM_PIN_PWDN    -1
#define CAM_PIN_RESET   -1
#define CAM_PIN_XCLK    10
#define CAM_PIN_SIOD    40
#define CAM_PIN_SIOC    39
#define CAM_PIN_D7      48
#define CAM_PIN_D6      11
#define CAM_PIN_D5      12
#define CAM_PIN_D4      14
#define CAM_PIN_D3      16
#define CAM_PIN_D2      18
#define CAM_PIN_D1      17
#define CAM_PIN_D0      15
#define CAM_PIN_VSYNC   38
#define CAM_PIN_HREF    47
#define CAM_PIN_PCLK    13

static bool cameraReady = false;

bool initCamera() {
    camera_config_t config;
    config.ledc_channel = LEDC_CHANNEL_0;
    config.ledc_timer = LEDC_TIMER_0;
    config.pin_d0 = CAM_PIN_D0;
    config.pin_d1 = CAM_PIN_D1;
    config.pin_d2 = CAM_PIN_D2;
    config.pin_d3 = CAM_PIN_D3;
    config.pin_d4 = CAM_PIN_D4;
    config.pin_d5 = CAM_PIN_D5;
    config.pin_d6 = CAM_PIN_D6;
    config.pin_d7 = CAM_PIN_D7;
    config.pin_xclk = CAM_PIN_XCLK;
    config.pin_pclk = CAM_PIN_PCLK;
    config.pin_vsync = CAM_PIN_VSYNC;
    config.pin_href = CAM_PIN_HREF;
    config.pin_sccb_sda = CAM_PIN_SIOD;
    config.pin_sccb_scl = CAM_PIN_SIOC;
    config.pin_pwdn = CAM_PIN_PWDN;
    config.pin_reset = CAM_PIN_RESET;
    config.xclk_freq_hz = 20000000;
    config.frame_size = FRAMESIZE_VGA;
    config.pixel_format = PIXFORMAT_JPEG;
    config.grab_mode = CAMERA_GRAB_LATEST;
    config.fb_location = CAMERA_FB_IN_PSRAM;
    config.jpeg_quality = 12;
    config.fb_count = 2;

    if (!psramFound()) {
        config.frame_size = FRAMESIZE_QVGA;
        config.fb_location = CAMERA_FB_IN_DRAM;
        config.fb_count = 1;
    }

    esp_err_t err = esp_camera_init(&config);
    if (err != ESP_OK) {
        Serial.printf("Camera init failed: 0x%x\n", err);
        return false;
    }

    // Basic sensor setup
    sensor_t* s = esp_camera_sensor_get();
    if (s) {
        s->set_brightness(s, 1);
        s->set_contrast(s, 1);
        s->set_saturation(s, 0);
        s->set_whitebal(s, 1);
        s->set_awb_gain(s, 1);
        s->set_exposure_ctrl(s, 1);
        s->set_gain_ctrl(s, 1);
        s->set_ae_level(s, -1);  // Reduce exposure by ~1 EV (range: -2 to +2)
        
        // Hardware orientation (disabled - using software transforms in cloud_ocr.h)
        s->set_vflip(s, 0);    // Vertical flip: 0=off, 1=on
        s->set_hmirror(s, 0);  // Horizontal mirror: 0=off, 1=on
    }

    cameraReady = true;
    return true;
}

/**
 * Capture JPEG frame
 */
camera_fb_t* captureJpeg() {
    if (!cameraReady) return nullptr;
    return esp_camera_fb_get();
}

void releaseFrame(camera_fb_t* fb) {
    if (fb) esp_camera_fb_return(fb);
}

/**
 * Capture and convert to grayscale
 */
bool captureGray(GrayImage& out) {
    camera_fb_t* fb = captureJpeg();
    if (!fb) return false;

    bool ok = false;
    
    if (fb->format == PIXFORMAT_JPEG) {
        // Decode JPEG to RGB then to gray
        size_t rgbLen = fb->width * fb->height * 3;
        uint8_t* rgb = (uint8_t*)heap_caps_malloc(rgbLen, MALLOC_CAP_SPIRAM);
        if (!rgb) rgb = (uint8_t*)malloc(rgbLen);
        
        if (rgb && fmt2rgb888(fb->buf, fb->len, PIXFORMAT_JPEG, rgb)) {
            if (out.alloc(fb->width, fb->height)) {
                for (int i = 0; i < fb->width * fb->height; i++) {
                    // Y = 0.299R + 0.587G + 0.114B
                    out.data[i] = (rgb[i*3] * 77 + rgb[i*3+1] * 150 + rgb[i*3+2] * 29) >> 8;
                }
                ok = true;
            }
        }
        if (rgb) free(rgb);
    }
    
    releaseFrame(fb);
    return ok;
}

#endif

