/**
 * i2c_comm.h - I2C slave communication with master
 * 
 * Protocol:
 *   Master writes: 2 bytes [current_slide (0-80), numpad_input (1-4)]
 *   Slave captures image, processes, returns result
 *   Master reads: 1 byte (result 0-80, or 0xFF for processing/error)
 */

#ifndef I2C_COMM_H
#define I2C_COMM_H

#include <Arduino.h>
#include <Wire.h>

// Default I2C config
#define I2C_ADDR    0x42
#define I2C_SDA     5
#define I2C_SCL     6

// State
static volatile uint8_t i2cCurrentSlide = 0;   // Byte 0: current slide (0-80)
static volatile uint8_t i2cNumpadInput = 0;    // Byte 1: numpad input (1-4)
static volatile uint8_t i2cResult = 0xFF;      // Result (0-80, 0xFF = not ready)
static volatile bool i2cPending = false;

void onI2CReceive(int len) {
    if (len >= 2) {
        // New 2-byte protocol
        i2cCurrentSlide = Wire.read();
        i2cNumpadInput = Wire.read();
        i2cResult = 0xFF;  // Mark as processing
        i2cPending = true;
        Serial.printf("[I2C] Received: slide=%d, numpad=%d\n", 
                      i2cCurrentSlide, i2cNumpadInput);
    } else if (len == 1) {
        // Legacy 1-byte protocol (backward compatibility)
        i2cCurrentSlide = Wire.read();
        i2cNumpadInput = 0;
        i2cResult = 0xFF;
        i2cPending = true;
        Serial.printf("[I2C] Received (legacy): req=%d\n", i2cCurrentSlide);
    }
}

void onI2CRequest() {
    Wire.write(i2cResult);
}

void initI2C(uint8_t addr = I2C_ADDR, int sda = I2C_SDA, int scl = I2C_SCL) {
    Wire.begin(addr, sda, scl, 400000);
    Wire.onReceive(onI2CReceive);
    Wire.onRequest(onI2CRequest);
    Serial.printf("I2C slave at 0x%02X (SDA=%d, SCL=%d)\n", addr, sda, scl);
}

bool hasI2CRequest() {
    return i2cPending;
}

/**
 * Get the current slide number from the request
 * @return Current slide (0-80)
 */
uint8_t getI2CCurrentSlide() {
    return i2cCurrentSlide;
}

/**
 * Get the numpad input from the request
 * @return Numpad input (1-4, or 0 for legacy single-byte request)
 */
uint8_t getI2CNumpadInput() {
    return i2cNumpadInput;
}

/**
 * Get the request (legacy - returns current slide for backward compatibility)
 * @return Current slide value
 */
uint8_t getI2CRequest() {
    i2cPending = false;
    return i2cCurrentSlide;
}

/**
 * Clear the pending flag (call after processing)
 */
void clearI2CRequest() {
    i2cPending = false;
}

/**
 * Set the result to be returned to master
 * @param val Result value (0-80, or 0xFF for error/not ready)
 */
void setI2CResult(uint8_t val) {
    i2cResult = val;
}

#endif
