/**
 * vision/cloud_ocr.h - Cloud OCR via Vercel API
 *
 * Sends images to the deployed Vercel OCR service and
 * extracts numbers from the right column of a scorecard.
 */

#ifndef VISION_CLOUD_OCR_H
#define VISION_CLOUD_OCR_H

#include <WiFi.h>
#include <HTTPClient.h>
#include <base64.h>
#include <ArduinoJson.h>
#include <esp_camera.h>
#include "image.h"
#include "transforms.h"

// Vercel OCR API endpoint
#define OCR_API_URL "hypercarousel-v0.vercel.app/api/ocr"

// Maximum numbers to detect
#define MAX_OCR_NUMBERS 10

struct CloudOCRResult
{
    int numbers[MAX_OCR_NUMBERS];
    int count;
    bool success;
    String error;

    CloudOCRResult() : count(0), success(false)
    {
        for (int i = 0; i < MAX_OCR_NUMBERS; i++)
            numbers[i] = 0;
    }
};

/**
 * Call Vercel OCR API with JPEG image
 */
inline CloudOCRResult cloudOCR(const uint8_t *jpegData, size_t jpegLen)
{
    CloudOCRResult result;

    if (!WiFi.isConnected())
    {
        result.error = "WiFi not connected";
        Serial.println("CloudOCR: WiFi not connected");
        return result;
    }

    Serial.printf("CloudOCR: Processing %d bytes JPEG\n", jpegLen);

    // Base64 encode the image
    String base64Image = base64::encode(jpegData, jpegLen);
    Serial.printf("CloudOCR: Base64 size: %d\n", base64Image.length());

    // Build request JSON
    JsonDocument doc;
    doc["image"] = base64Image;
    String requestJson;
    serializeJson(doc, requestJson);

    // Make API request
    HTTPClient http;

    http.begin(OCR_API_URL);
    http.addHeader("Content-Type", "application/json");
    http.setTimeout(30000); // 30 second timeout

    Serial.println("CloudOCR: Sending request to Vercel API...");
    int httpCode = http.POST(requestJson);

    if (httpCode != 200)
    {
        result.error = "HTTP error: " + String(httpCode);
        Serial.printf("CloudOCR: HTTP error %d\n", httpCode);
        if (httpCode > 0)
        {
            String response = http.getString();
            Serial.println(response.substring(0, 500));
        }
        http.end();
        return result;
    }

    String response = http.getString();
    http.end();

    Serial.printf("CloudOCR: Response size: %d\n", response.length());

    // Print full response for debugging
    Serial.println("===== API Response =====");
    Serial.println(response);
    Serial.println("========================");

    // Parse JSON response
    JsonDocument responseDoc;
    DeserializationError parseError = deserializeJson(responseDoc, response);

    if (parseError)
    {
        result.error = "JSON parse error: " + String(parseError.c_str());
        Serial.printf("CloudOCR: JSON parse error: %s\n", parseError.c_str());
        return result;
    }

    // Print fullText first for debugging
    if (responseDoc["fullText"].is<const char *>())
    {
        String fullText = responseDoc["fullText"].as<String>();
        Serial.println("===== OCR Full Text =====");
        Serial.println(fullText);
        Serial.println("=========================");
    }

    // Check for API error
    if (responseDoc["error"].is<const char *>())
    {
        result.error = responseDoc["error"].as<String>();
        Serial.printf("CloudOCR: API error: %s\n", result.error.c_str());
        if (responseDoc["details"].is<const char *>())
        {
            Serial.printf("CloudOCR: Details: %s\n", responseDoc["details"].as<String>().c_str());
        }
        return result;
    }

    // Print debug info
    if (responseDoc["debug"]["annotationsCount"].is<int>())
    {
        Serial.printf("CloudOCR: Vision API found %d text annotations\n",
                      responseDoc["debug"]["annotationsCount"].as<int>());
    }

    JsonArray allNums = responseDoc["debug"]["allDetectedNumbers"].as<JsonArray>();
    if (!allNums.isNull() && allNums.size() > 0)
    {
        Serial.printf("CloudOCR: All detected numbers (%d):\n", allNums.size());
        for (JsonVariant n : allNums)
        {
            Serial.printf("  %d at (%d, %d)\n",
                          n["value"].as<int>(), n["x"].as<int>(), n["y"].as<int>());
        }
    }

    // Extract numbers array
    JsonArray numbersArray = responseDoc["numbers"].as<JsonArray>();

    if (numbersArray.isNull() || numbersArray.size() == 0)
    {
        result.error = "No numbers extracted from image";
        Serial.println("CloudOCR: No numbers in final result!");
        Serial.println("CloudOCR: The Vision API may have detected text but no valid numbers were extracted.");
        return result;
    }

    // Copy numbers to result
    for (JsonVariant num : numbersArray)
    {
        if (result.count < MAX_OCR_NUMBERS)
        {
            result.numbers[result.count] = num.as<int>();
            result.count++;
        }
    }

    result.success = result.count > 0;

    Serial.printf("CloudOCR: Result: %d numbers:", result.count);
    for (int i = 0; i < result.count; i++)
    {
        Serial.printf(" %d", result.numbers[i]);
    }
    Serial.println();

    // Also log full text if available
    if (responseDoc["fullText"].is<const char *>())
    {
        String fullText = responseDoc["fullText"].as<String>();
        Serial.printf("CloudOCR: Full text: %s\n", fullText.c_str());
    }

    return result;
}

/**
 * Apply contrast enhancement for better OCR
 */
inline void enhanceForOCR(GrayImage &img)
{
    // Find min/max for contrast stretch
    uint8_t minVal = 255, maxVal = 0;
    for (int i = 0; i < img.width * img.height; i++)
    {
        if (img.data[i] < minVal)
            minVal = img.data[i];
        if (img.data[i] > maxVal)
            maxVal = img.data[i];
    }

    // Avoid division by zero
    if (maxVal <= minVal)
        return;

    float scale = 255.0f / (maxVal - minVal);

    // Contrast stretch
    for (int i = 0; i < img.width * img.height; i++)
    {
        int val = (int)((img.data[i] - minVal) * scale);
        img.data[i] = (uint8_t)(val < 0 ? 0 : (val > 255 ? 255 : val));
    }

    Serial.printf("enhanceForOCR: stretched [%d-%d] -> [0-255]\n", minVal, maxVal);
}

/**
 * Convert grayscale image to JPEG
 * Returns allocated buffer (caller must free) and sets jpegLen
 */
inline uint8_t *grayToJpeg(const GrayImage &gray, size_t *jpegLen, int quality = 95)
{
    // Convert grayscale to RGB888 (just duplicate gray value 3x)
    size_t rgbLen = gray.width * gray.height * 3;
    uint8_t *rgb = (uint8_t *)heap_caps_malloc(rgbLen, MALLOC_CAP_SPIRAM);
    if (!rgb)
        rgb = (uint8_t *)malloc(rgbLen);
    if (!rgb)
    {
        Serial.println("grayToJpeg: RGB allocation failed");
        return nullptr;
    }

    for (int i = 0; i < gray.width * gray.height; i++)
    {
        rgb[i * 3] = gray.data[i];
        rgb[i * 3 + 1] = gray.data[i];
        rgb[i * 3 + 2] = gray.data[i];
    }

    // Encode to JPEG
    uint8_t *jpegBuf = nullptr;
    bool ok = fmt2jpg(rgb, rgbLen, gray.width, gray.height, PIXFORMAT_RGB888, quality, &jpegBuf, jpegLen);

    free(rgb);

    if (!ok || !jpegBuf)
    {
        Serial.println("grayToJpeg: JPEG encoding failed");
        return nullptr;
    }

    Serial.printf("grayToJpeg: %dx%d -> %d bytes JPEG\n", gray.width, gray.height, *jpegLen);
    return jpegBuf;
}

/**
 * Capture image, apply transforms, and run Cloud OCR
 * Transforms: 90° CCW rotation + horizontal flip
 */
inline CloudOCRResult captureAndOCR()
{
    CloudOCRResult result;

    // Capture JPEG
    camera_fb_t *fb = esp_camera_fb_get();
    if (!fb)
    {
        result.error = "Camera capture failed";
        Serial.println("CloudOCR: Camera capture failed");
        return result;
    }

    if (fb->format != PIXFORMAT_JPEG)
    {
        result.error = "Not JPEG format";
        esp_camera_fb_return(fb);
        return result;
    }

    Serial.printf("CloudOCR: Captured %dx%d JPEG (%d bytes)\n", fb->width, fb->height, fb->len);

    // Decode JPEG to RGB then to grayscale
    size_t rgbLen = fb->width * fb->height * 3;
    uint8_t *rgb = (uint8_t *)heap_caps_malloc(rgbLen, MALLOC_CAP_SPIRAM);
    if (!rgb)
        rgb = (uint8_t *)malloc(rgbLen);

    if (!rgb)
    {
        result.error = "RGB allocation failed";
        esp_camera_fb_return(fb);
        return result;
    }

    if (!fmt2rgb888(fb->buf, fb->len, PIXFORMAT_JPEG, rgb))
    {
        result.error = "JPEG decode failed";
        free(rgb);
        esp_camera_fb_return(fb);
        return result;
    }

    // Convert to grayscale
    GrayImage original;
    if (!original.alloc(fb->width, fb->height))
    {
        result.error = "Grayscale allocation failed";
        free(rgb);
        esp_camera_fb_return(fb);
        return result;
    }

    for (int i = 0; i < fb->width * fb->height; i++)
    {
        // Y = 0.299R + 0.587G + 0.114B
        original.data[i] = (rgb[i * 3] * 77 + rgb[i * 3 + 1] * 150 + rgb[i * 3 + 2] * 29) >> 8;
    }
    free(rgb);
    esp_camera_fb_return(fb);

    Serial.printf("CloudOCR: Decoded to grayscale %dx%d\n", original.width, original.height);

    // Apply transforms: 90° CCW rotation + horizontal flip
    GrayImage rotated;
    if (!rotate90CCW(original, rotated))
    {
        result.error = "Rotation failed";
        return result;
    }
    original.release();
    Serial.printf("CloudOCR: Rotated 90° CCW -> %dx%d\n", rotated.width, rotated.height);

    GrayImage flipped;
    if (!flipH(rotated, flipped))
    {
        result.error = "Flip failed";
        return result;
    }
    rotated.release();
    Serial.printf("CloudOCR: Flipped horizontally -> %dx%d\n", flipped.width, flipped.height);

    // Enhance contrast for better OCR
    enhanceForOCR(flipped);

    // Encode back to JPEG with high quality
    size_t jpegLen = 0;
    uint8_t *jpegBuf = grayToJpeg(flipped, &jpegLen, 95);
    flipped.release();

    if (!jpegBuf)
    {
        result.error = "JPEG encoding failed";
        return result;
    }

    // Run OCR on processed image
    result = cloudOCR(jpegBuf, jpegLen);

    free(jpegBuf);
    return result;
}

#endif
