/**
 * vision/detector.h - L-mark corner detection and row analysis
 */

#ifndef VISION_DETECTOR_H
#define VISION_DETECTOR_H

#include "image.h"
#include <algorithm>

struct Point2i {
    int x, y;
    Point2i() : x(0), y(0) {}
    Point2i(int _x, int _y) : x(_x), y(_y) {}
};

// L-mark orientations (corner position relative to the L shape)
// TL: corner at top-left, arms go right and down
// TR: corner at top-right, arms go left and down
// BL: corner at bottom-left, arms go right and up
// BR: corner at bottom-right, arms go left and up
enum LMarkType { LMARK_TL = 0, LMARK_TR, LMARK_BL, LMARK_BR };

struct LMark {
    Point2i pos;
    float score;
    bool found;
    LMark() : pos(), score(0), found(false) {}
};

struct Quad {
    Point2i tl, tr, bl, br;  // corners
    bool valid;
    Quad() : valid(false) {}
};

struct RowCluster {
    int yStart;
    int yEnd;
    int centerY;
};

struct DetectionResult {
    Quad quad;
    int rowCount;           // number of detected rows (1-5)
    RowCluster rows[5];     // up to 5 rows
    bool success;
    
    DetectionResult() : rowCount(0), success(false) {}
};

/**
 * Score L-mark at position for given orientation
 * Returns score > 0 if found, 0 if not
 */
inline float scoreLMark(const GrayImage& img, int cx, int cy, LMarkType type, 
                        int minArm = 8, int maxArm = 50, uint8_t darkThresh = 80) {
    // Direction vectors based on orientation
    int hDir, vDir;  // horizontal and vertical arm directions
    switch (type) {
        case LMARK_TL: hDir = 1;  vDir = 1;  break;  // right, down
        case LMARK_TR: hDir = -1; vDir = 1;  break;  // left, down
        case LMARK_BL: hDir = 1;  vDir = -1; break;  // right, up
        case LMARK_BR: hDir = -1; vDir = -1; break;  // left, up
    }
    
    // Check corner is dark
    if (img.get(cx, cy) > darkThresh) return 0;
    
    // Measure horizontal arm length
    int hLen = 0;
    for (int i = 1; i <= maxArm; i++) {
        int x = cx + hDir * i;
        if (img.get(x, cy) < darkThresh) hLen++;
        else break;
    }
    
    // Measure vertical arm length
    int vLen = 0;
    for (int i = 1; i <= maxArm; i++) {
        int y = cy + vDir * i;
        if (img.get(cx, y) < darkThresh) vLen++;
        else break;
    }
    
    if (hLen < minArm || vLen < minArm) return 0;
    
    // Check inside of L is light (not filled)
    int checkSize = std::min(hLen, vLen) / 2;
    int lightCount = 0, totalCount = 0;
    for (int dy = 1; dy <= checkSize; dy++) {
        for (int dx = 1; dx <= checkSize; dx++) {
            int x = cx + hDir * dx;
            int y = cy + vDir * dy;
            if (img.get(x, y) > darkThresh + 40) lightCount++;
            totalCount++;
        }
    }
    
    if (totalCount > 0 && lightCount < totalCount * 0.5f) return 0;
    
    // Score based on arm lengths and balance
    float score = (hLen + vLen) / 2.0f;
    float ratio = (float)std::min(hLen, vLen) / std::max(hLen, vLen);
    return score * (0.5f + 0.5f * ratio);
}

/**
 * Find L-mark in a region
 */
inline LMark findLMark(const GrayImage& img, int x0, int y0, int x1, int y1, 
                       LMarkType type, int step = 3) {
    LMark best;
    
    for (int y = y0; y < y1; y += step) {
        for (int x = x0; x < x1; x += step) {
            float score = scoreLMark(img, x, y, type);
            if (score > best.score) {
                best.score = score;
                best.pos = Point2i(x, y);
            }
        }
    }
    
    // Refine with finer search
    if (best.score > 0) {
        int rx = best.pos.x, ry = best.pos.y;
        for (int dy = -step; dy <= step; dy++) {
            for (int dx = -step; dx <= step; dx++) {
                float score = scoreLMark(img, rx + dx, ry + dy, type);
                if (score > best.score) {
                    best.score = score;
                    best.pos = Point2i(rx + dx, ry + dy);
                }
            }
        }
        best.found = true;
    }
    
    return best;
}

/**
 * Detect all 4 L-marks and form quad
 */
inline Quad detectQuad(const GrayImage& img) {
    Quad q;
    
    int w = img.width, h = img.height;
    
    // Crop regions: skip left 20%, bottom 23%
    int cropLeft = w * 20 / 100;
    int cropBottom = h * 23 / 100;
    int useW = w - cropLeft;
    int useH = h - cropBottom;
    
    // Search regions for each corner (in the usable area)
    int midX = cropLeft + useW / 2;
    int midY = useH / 2;
    
    // TL: top-left quadrant of usable area
    LMark tl = findLMark(img, cropLeft, 0, midX, midY, LMARK_TL);
    
    // TR: top-right quadrant
    LMark tr = findLMark(img, midX, 0, w, midY, LMARK_TR);
    
    // BL: bottom-left quadrant (above crop line)
    LMark bl = findLMark(img, cropLeft, midY, midX, useH, LMARK_BL);
    
    // BR: bottom-right quadrant (above crop line)
    LMark br = findLMark(img, midX, midY, w, useH, LMARK_BR);
    
    if (tl.found && tr.found && bl.found && br.found) {
        q.tl = tl.pos;
        q.tr = tr.pos;
        q.bl = bl.pos;
        q.br = br.pos;
        q.valid = true;
    }
    
    return q;
}

/**
 * Analyze right portion of quad for dark pixel clusters (rows)
 */
inline DetectionResult analyzeRows(const GrayImage& img, const Quad& q, uint8_t darkThresh = 100) {
    DetectionResult result;
    
    if (!q.valid) return result;
    
    // Get bounds of right portion (rightmost 40% of quad width)
    int quadLeft = std::min(q.tl.x, q.bl.x);
    int quadRight = std::max(q.tr.x, q.br.x);
    int quadTop = std::min(q.tl.y, q.tr.y);
    int quadBottom = std::max(q.bl.y, q.br.y);
    
    int quadW = quadRight - quadLeft;
    int quadH = quadBottom - quadTop;
    
    // Right portion: last 40% horizontally
    int analyzeLeft = quadLeft + quadW * 60 / 100;
    int analyzeRight = quadRight - 5;  // small margin
    
    // Count dark pixels per Y row
    int* darkCount = new int[quadH]();
    int maxCount = 0;
    
    for (int y = quadTop; y < quadBottom; y++) {
        int count = 0;
        for (int x = analyzeLeft; x < analyzeRight; x++) {
            if (img.get(x, y) < darkThresh) count++;
        }
        darkCount[y - quadTop] = count;
        if (count > maxCount) maxCount = count;
    }
    
    // Find clusters of rows with significant dark pixels
    int threshold = maxCount / 3;  // at least 1/3 of max to count as a row
    
    bool inCluster = false;
    int clusterStart = 0;
    
    for (int y = 0; y < quadH; y++) {
        bool isDark = darkCount[y] > threshold;
        
        if (isDark && !inCluster) {
            // Start new cluster
            inCluster = true;
            clusterStart = y;
        } else if (!isDark && inCluster) {
            // End cluster
            if (result.rowCount < 5) {
                result.rows[result.rowCount].yStart = quadTop + clusterStart;
                result.rows[result.rowCount].yEnd = quadTop + y;
                result.rows[result.rowCount].centerY = quadTop + (clusterStart + y) / 2;
                result.rowCount++;
            }
            inCluster = false;
        }
    }
    
    // Handle cluster at end
    if (inCluster && result.rowCount < 5) {
        result.rows[result.rowCount].yStart = quadTop + clusterStart;
        result.rows[result.rowCount].yEnd = quadTop + quadH;
        result.rows[result.rowCount].centerY = quadTop + (clusterStart + quadH) / 2;
        result.rowCount++;
    }
    
    delete[] darkCount;
    
    result.quad = q;
    result.success = (result.rowCount >= 1 && result.rowCount <= 5);
    
    return result;
}

/**
 * Full detection pipeline
 */
inline DetectionResult detectMarks(const GrayImage& img) {
    Quad q = detectQuad(img);
    return analyzeRows(img, q);
}

/**
 * Process detection and return result code (row count or specific row)
 */
inline uint8_t processDetection(const GrayImage& img, uint8_t requestId) {
    DetectionResult det = detectMarks(img);
    
    if (!det.success) {
        return 0xFF;  // error
    }
    
    // Return row count for now
    // TODO: expand based on requestId to return specific data
    return det.rowCount;
}

#endif
