/**
 * vision/image.h - Grayscale image buffer with PSRAM support
 */

#ifndef VISION_IMAGE_H
#define VISION_IMAGE_H

#include <Arduino.h>
#include <esp_heap_caps.h>

struct GrayImage {
    uint8_t* data = nullptr;
    int width = 0;
    int height = 0;
    bool owned = false;

    GrayImage() = default;
    ~GrayImage() { release(); }

    // No copy
    GrayImage(const GrayImage&) = delete;
    GrayImage& operator=(const GrayImage&) = delete;

    // Move OK
    GrayImage(GrayImage&& o) noexcept {
        data = o.data; width = o.width; height = o.height; owned = o.owned;
        o.data = nullptr; o.owned = false;
    }
    GrayImage& operator=(GrayImage&& o) noexcept {
        if (this != &o) {
            release();
            data = o.data; width = o.width; height = o.height; owned = o.owned;
            o.data = nullptr; o.owned = false;
        }
        return *this;
    }

    bool alloc(int w, int h) {
        release();
        data = (uint8_t*)heap_caps_malloc(w * h, MALLOC_CAP_SPIRAM | MALLOC_CAP_8BIT);
        if (!data) {
            data = (uint8_t*)malloc(w * h);
        }
        if (!data) return false;
        width = w;
        height = h;
        owned = true;
        return true;
    }

    void release() {
        if (owned && data) free(data);
        data = nullptr;
        width = height = 0;
        owned = false;
    }

    void clear(uint8_t val = 0) {
        if (data) memset(data, val, width * height);
    }

    inline uint8_t get(int x, int y) const {
        if (x < 0 || x >= width || y < 0 || y >= height) return 0;
        return data[y * width + x];
    }

    inline void set(int x, int y, uint8_t val) {
        if (x >= 0 && x < width && y >= 0 && y < height)
            data[y * width + x] = val;
    }

    inline uint8_t at(int x, int y) const {
        return data[y * width + x];
    }

    // Bilinear sample for sub-pixel access
    uint8_t sample(float fx, float fy) const {
        if (fx < 0 || fx >= width - 1 || fy < 0 || fy >= height - 1) return 0;
        int x0 = (int)fx, y0 = (int)fy;
        float dx = fx - x0, dy = fy - y0;
        float p00 = at(x0, y0), p10 = at(x0 + 1, y0);
        float p01 = at(x0, y0 + 1), p11 = at(x0 + 1, y0 + 1);
        float top = p00 * (1 - dx) + p10 * dx;
        float bot = p01 * (1 - dx) + p11 * dx;
        return (uint8_t)(top * (1 - dy) + bot * dy);
    }

    size_t size() const { return width * height; }
    bool valid() const { return data != nullptr && width > 0 && height > 0; }
};

#endif

