/**
 * vision/transforms.h - Image transformations (rotate, keystone)
 */

#ifndef VISION_TRANSFORMS_H
#define VISION_TRANSFORMS_H

#include "image.h"

/**
 * Rotate image 90 degrees counter-clockwise
 * Input WxH -> Output HxW
 */
inline bool rotate90CCW(const GrayImage& src, GrayImage& dst) {
    if (!dst.alloc(src.height, src.width)) return false;
    
    for (int y = 0; y < src.height; y++) {
        for (int x = 0; x < src.width; x++) {
            // CCW: (x,y) -> (y, W-1-x)
            int nx = y;
            int ny = src.width - 1 - x;
            dst.data[ny * dst.width + nx] = src.at(x, y);
        }
    }
    return true;
}

/**
 * Rotate image 90 degrees clockwise
 * Input WxH -> Output HxW
 */
inline bool rotate90CW(const GrayImage& src, GrayImage& dst) {
    if (!dst.alloc(src.height, src.width)) return false;
    
    for (int y = 0; y < src.height; y++) {
        for (int x = 0; x < src.width; x++) {
            // CW: (x,y) -> (H-1-y, x)
            int nx = src.height - 1 - y;
            int ny = x;
            dst.data[ny * dst.width + nx] = src.at(x, y);
        }
    }
    return true;
}

/**
 * Horizontal keystone correction
 * 
 * Corrects perspective where left/right sides have different heights.
 * 
 * @param amount Range -1.0 to +1.0
 *   Positive: right side is closer (taller in image) -> compress right
 *   Negative: left side is closer (taller in image) -> compress left
 *   Zero: no correction
 */
inline bool keystoneH(const GrayImage& src, GrayImage& dst, float amount) {
    if (!dst.alloc(src.width, src.height)) return false;
    dst.clear(128);  // gray fill for out-of-bounds
    
    float cy = src.height / 2.0f;
    
    for (int x = 0; x < src.width; x++) {
        // t: 0 at left, 1 at right
        float t = (float)x / (src.width - 1);
        // scale factor for this column
        // positive amount -> right columns get compressed (scale > 1 samples from larger range)
        float scale = 1.0f + amount * (2.0f * t - 1.0f);
        
        for (int y = 0; y < src.height; y++) {
            float dy = y - cy;
            float srcY = cy + dy / scale;
            if (srcY >= 0 && srcY < src.height - 1) {
                dst.set(x, y, src.sample((float)x, srcY));
            }
        }
    }
    return true;
}

/**
 * Rotate 180 degrees: (x,y) -> (W-1-x, H-1-y)
 */
inline bool rotate180(const GrayImage& src, GrayImage& dst) {
    if (!dst.alloc(src.width, src.height)) return false;
    
    int total = src.width * src.height;
    for (int i = 0; i < total; i++) {
        dst.data[i] = src.data[total - 1 - i];
    }
    return true;
}

/**
 * Vertical flip
 */
inline bool flipV(const GrayImage& src, GrayImage& dst) {
    if (!dst.alloc(src.width, src.height)) return false;
    
    for (int y = 0; y < src.height; y++) {
        int srcY = src.height - 1 - y;
        memcpy(dst.data + y * src.width, src.data + srcY * src.width, src.width);
    }
    return true;
}

/**
 * Horizontal flip
 */
inline bool flipH(const GrayImage& src, GrayImage& dst) {
    if (!dst.alloc(src.width, src.height)) return false;
    
    for (int y = 0; y < src.height; y++) {
        for (int x = 0; x < src.width; x++) {
            dst.data[y * src.width + x] = src.at(src.width - 1 - x, y);
        }
    }
    return true;
}

/**
 * Vertical keystone correction (for reference)
 */
inline bool keystoneV(const GrayImage& src, GrayImage& dst, float amount) {
    if (!dst.alloc(src.width, src.height)) return false;
    dst.clear(128);
    
    float cx = src.width / 2.0f;
    
    for (int y = 0; y < src.height; y++) {
        float t = (float)y / (src.height - 1);
        float scale = 1.0f + amount * (2.0f * t - 1.0f);
        
        for (int x = 0; x < src.width; x++) {
            float dx = x - cx;
            float srcX = cx + dx / scale;
            if (srcX >= 0 && srcX < src.width - 1) {
                dst.set(x, y, src.sample(srcX, (float)y));
            }
        }
    }
    return true;
}

#endif

