/**
 * wifi_server.h - Minimal WiFi debug server
 */

#ifndef WIFI_SERVER_H
#define WIFI_SERVER_H

#include <WiFi.h>
#include <WebServer.h>
#include "credentials.h"
#include "camera_config.h"
#include "vision/image.h"
#include "vision/transforms.h"
#include "vision/detector.h"
#include "vision/cloud_ocr.h"

static WebServer server(80);
static float keystoneVal = -0.44f; // calibrated default
static int orientMode = 2;         // CCW + FlipH

// Minimal HTML - no special chars, minimal CSS
static const char HTML_PAGE[] PROGMEM = R"rawliteral(
<!DOCTYPE html>
<html>
<head>
<title>ESP32-S3 CAM</title>
<meta name="viewport" content="width=device-width,initial-scale=1">
<style>
body{font-family:sans-serif;margin:20px;background:#f0f0f0}
h1{margin:0 0 10px}
.box{background:#fff;padding:15px;margin:10px 0;border:1px solid #ccc}
img,canvas{max-width:100%;border:1px solid #999;background:#eee}
button{padding:8px 16px;margin:5px;cursor:pointer}
.row{display:flex;flex-wrap:wrap;gap:10px}
.col{flex:1;min-width:300px}
input[type=range]{width:200px}
#log{font-family:monospace;font-size:12px;background:#222;color:#0f0;padding:10px;height:80px;overflow:auto}
</style>
</head>
<body>
<h1>ESP32-S3 CAM Debug</h1>

<div class="box">
<button onclick="capture()">Capture</button>
<button onclick="toggleStream()">Stream</button>
<button onclick="detect()">Detect</button>
<button onclick="cloudOCR()" style="background:#4CAF50;color:white">Vercel OCR</button>
<span id="status">Ready</span>
</div>

<div class="row">
<div class="col box">
<h3>Camera</h3>
<img id="cam" src="/capture">
</div>
<div class="col box">
<h3>Processed</h3>
<canvas id="proc"></canvas>
</div>
</div>

<div class="box">
<h3>Orientation (test modes 0-7)</h3>
<button onclick="setOrient(0)">0: CCW</button>
<button onclick="setOrient(1)">1: CCW+180</button>
<button onclick="setOrient(2)">2: CCW+FlipH</button>
<button onclick="setOrient(3)">3: CCW+FlipV</button>
<button onclick="setOrient(4)">4: CW</button>
<button onclick="setOrient(5)">5: CW+180</button>
<button onclick="setOrient(6)">6: CW+FlipH</button>
<button onclick="setOrient(7)">7: CW+FlipV</button>
<span id="orientVal">Mode: 2</span>
</div>

<div class="box">
<h3>Keystone</h3>
<input type="range" id="ks" min="-100" max="100" value="-44" oninput="setKs(this.value)">
<span id="ksVal">-0.44</span>
</div>

<div class="box">
<h3>Log</h3>
<div id="log"></div>
</div>

<script>
var streaming=false,interval=null;

function log(m){
  var d=document.getElementById("log");
  d.innerHTML+=m+"<br>";
  d.scrollTop=d.scrollHeight;
}

function status(m){document.getElementById("status").innerText=m;}

function capture(){
  document.getElementById("cam").src="/capture?t="+Date.now();
  fetchProc();
}

function toggleStream(){
  if(streaming){clearInterval(interval);streaming=false;status("Stopped");}
  else{interval=setInterval(capture,200);streaming=true;status("Streaming");}
}

function setKs(v){
  var r=v/100;
  document.getElementById("ksVal").innerText=r.toFixed(2);
  fetch("/keystone?v="+r).then(function(){fetchProc();});
}

function setOrient(m){
  document.getElementById("orientVal").innerText="Mode: "+m;
  fetch("/orient?m="+m).then(function(){fetchProc();});
}

async function fetchProc(){
  try{
    var r=await fetch("/processed?t="+Date.now());
    var w=parseInt(r.headers.get("X-Width")||0);
    var h=parseInt(r.headers.get("X-Height")||0);
    var buf=new Uint8Array(await r.arrayBuffer());
    if(w>0&&h>0&&buf.length>=w*h){
      var c=document.getElementById("proc");
      c.width=w;c.height=h;
      var ctx=c.getContext("2d");
      var img=ctx.createImageData(w,h);
      for(var i=0;i<w*h;i++){
        img.data[i*4]=buf[i];
        img.data[i*4+1]=buf[i];
        img.data[i*4+2]=buf[i];
        img.data[i*4+3]=255;
      }
      ctx.putImageData(img,0,0);
    }
  }catch(e){log("Error: "+e);}
}

async function detect(){
  log("Detecting...");
  try{
    var r=await fetch("/detect?t="+Date.now());
    var w=parseInt(r.headers.get("X-Width")||0);
    var h=parseInt(r.headers.get("X-Height")||0);
    var rows=r.headers.get("X-Rows")||"0";
    var quad=r.headers.get("X-Quad")||"";
    var buf=new Uint8Array(await r.arrayBuffer());
    if(w>0&&h>0&&buf.length>=w*h){
      var c=document.getElementById("proc");
      c.width=w;c.height=h;
      var ctx=c.getContext("2d");
      var img=ctx.createImageData(w,h);
      for(var i=0;i<w*h;i++){
        img.data[i*4]=buf[i];
        img.data[i*4+1]=buf[i];
        img.data[i*4+2]=buf[i];
        img.data[i*4+3]=255;
      }
      ctx.putImageData(img,0,0);
      // Draw quad overlay
      if(quad){
        var pts=quad.split(",").map(Number);
        if(pts.length==8){
          ctx.strokeStyle="#00ff00";
          ctx.lineWidth=2;
          ctx.beginPath();
          ctx.moveTo(pts[0],pts[1]);
          ctx.lineTo(pts[2],pts[3]);
          ctx.lineTo(pts[6],pts[7]);
          ctx.lineTo(pts[4],pts[5]);
          ctx.closePath();
          ctx.stroke();
          // Draw corner markers
          ctx.fillStyle="#ff0000";
          for(var i=0;i<8;i+=2){
            ctx.beginPath();
            ctx.arc(pts[i],pts[i+1],5,0,Math.PI*2);
            ctx.fill();
          }
        }
      }
      log("Rows detected: "+rows+" Quad: "+quad);
    }
  }catch(e){log("Error: "+e);}
}

async function cloudOCR(){
  status("Running Vercel OCR...");
  log("Calling Vercel OCR API...");
  try{
    var r=await fetch("/cloudocr?t="+Date.now());
    var json=await r.json();
    if(json.success){
      log("Vercel OCR SUCCESS: "+json.numbers.join(", "));
      status("Found: "+json.numbers.join(", "));
    }else{
      log("Vercel OCR FAILED: "+json.error);
      status("Error: "+json.error);
    }
  }catch(e){log("Error: "+e);status("Error");}
}

capture();
</script>
</body>
</html>
)rawliteral";

void handleRoot()
{
  server.send(200, "text/html", HTML_PAGE);
}

void handleCapture()
{
  camera_fb_t *fb = captureJpeg();
  if (!fb)
  {
    server.send(500, "text/plain", "capture failed");
    return;
  }
  server.sendHeader("Cache-Control", "no-cache");
  server.setContentLength(fb->len);
  server.send(200, "image/jpeg", "");
  server.client().write(fb->buf, fb->len);
  releaseFrame(fb);
}

void handleProcessed()
{
  GrayImage gray, step1, step2, corrected;

  if (!captureGray(gray))
  {
    server.send(500, "text/plain", "capture failed");
    return;
  }

  // Step 1: 90 degree rotation (CCW for modes 0-3, CW for modes 4-7)
  bool ok;
  if (orientMode < 4)
  {
    ok = rotate90CCW(gray, step1);
  }
  else
  {
    ok = rotate90CW(gray, step1);
  }
  gray.release();
  if (!ok)
  {
    server.send(500, "text/plain", "rotate failed");
    return;
  }

  // Step 2: additional transform based on mode
  int subMode = orientMode % 4;
  switch (subMode)
  {
  case 0: // no additional transform
    step2 = std::move(step1);
    break;
  case 1: // +180
    ok = rotate180(step1, step2);
    step1.release();
    break;
  case 2: // +flipH
    ok = flipH(step1, step2);
    step1.release();
    break;
  case 3: // +flipV
    ok = flipV(step1, step2);
    step1.release();
    break;
  }
  if (!ok)
  {
    server.send(500, "text/plain", "transform failed");
    return;
  }

  // Apply keystone
  GrayImage *output = &step2;
  if (keystoneVal != 0.0f)
  {
    if (keystoneH(step2, corrected, keystoneVal))
    {
      step2.release();
      output = &corrected;
    }
  }

  server.sendHeader("X-Width", String(output->width));
  server.sendHeader("X-Height", String(output->height));
  server.sendHeader("Cache-Control", "no-cache");
  server.setContentLength(output->size());
  server.send(200, "application/octet-stream", "");
  server.client().write(output->data, output->size());
}

void handleKeystone()
{
  if (server.hasArg("v"))
  {
    keystoneVal = server.arg("v").toFloat();
    if (keystoneVal < -1.0f)
      keystoneVal = -1.0f;
    if (keystoneVal > 1.0f)
      keystoneVal = 1.0f;
  }
  server.send(200, "text/plain", String(keystoneVal));
}

void handleOrient()
{
  if (server.hasArg("m"))
  {
    orientMode = server.arg("m").toInt();
    if (orientMode < 0)
      orientMode = 0;
    if (orientMode > 7)
      orientMode = 7;
  }
  server.send(200, "text/plain", String(orientMode));
}

void handleDetect()
{
  GrayImage gray, step1, step2, corrected;

  if (!captureGray(gray))
  {
    server.send(500, "text/plain", "capture failed");
    return;
  }

  // Apply same transforms as handleProcessed (mode 2 = CCW + FlipH)
  bool ok = rotate90CCW(gray, step1);
  gray.release();
  if (!ok)
  {
    server.send(500, "text/plain", "rotate failed");
    return;
  }

  ok = flipH(step1, step2);
  step1.release();
  if (!ok)
  {
    server.send(500, "text/plain", "flip failed");
    return;
  }

  // Apply keystone
  ok = keystoneH(step2, corrected, keystoneVal);
  step2.release();
  if (!ok)
  {
    server.send(500, "text/plain", "keystone failed");
    return;
  }

  // Run detection
  DetectionResult det = detectMarks(corrected);

  // Build quad string: tl.x,tl.y,tr.x,tr.y,bl.x,bl.y,br.x,br.y
  String quadStr = "";
  if (det.quad.valid)
  {
    quadStr = String(det.quad.tl.x) + "," + String(det.quad.tl.y) + "," +
              String(det.quad.tr.x) + "," + String(det.quad.tr.y) + "," +
              String(det.quad.bl.x) + "," + String(det.quad.bl.y) + "," +
              String(det.quad.br.x) + "," + String(det.quad.br.y);
  }

  server.sendHeader("X-Width", String(corrected.width));
  server.sendHeader("X-Height", String(corrected.height));
  server.sendHeader("X-Rows", String(det.rowCount));
  server.sendHeader("X-Quad", quadStr);
  server.sendHeader("Cache-Control", "no-cache");
  server.setContentLength(corrected.size());
  server.send(200, "application/octet-stream", "");
  server.client().write(corrected.data, corrected.size());
}

void handleStatus()
{
  String json = "{\"keystone\":" + String(keystoneVal) + "}";
  server.send(200, "application/json", json);
}

void handleCloudOCR()
{
  Serial.println("Cloud OCR request...");

  CloudOCRResult result = captureAndOCR();

  String json = "{";
  json += "\"success\":" + String(result.success ? "true" : "false") + ",";
  json += "\"count\":" + String(result.count) + ",";
  json += "\"numbers\":[";
  for (int i = 0; i < result.count; i++)
  {
    if (i > 0)
      json += ",";
    json += String(result.numbers[i]);
  }
  json += "],";
  json += "\"error\":\"" + result.error + "\"";
  json += "}";

  server.send(200, "application/json", json);
}

void initWiFi()
{
  Serial.print("Connecting to WiFi");
  WiFi.begin(WIFI_SSID, WIFI_PASSWORD);

  int tries = 0;
  while (WiFi.status() != WL_CONNECTED && tries < 30)
  {
    delay(500);
    Serial.print(".");
    tries++;
  }

  if (WiFi.status() != WL_CONNECTED)
  {
    Serial.println(" FAILED");
    return;
  }

  Serial.println(" OK");
  Serial.print("IP: ");
  Serial.println(WiFi.localIP());

  server.on("/", handleRoot);
  server.on("/capture", handleCapture);
  server.on("/processed", handleProcessed);
  server.on("/detect", handleDetect);
  server.on("/keystone", handleKeystone);
  server.on("/orient", handleOrient);
  server.on("/status", handleStatus);
  server.on("/cloudocr", handleCloudOCR);
  server.begin();

  Serial.printf("Server: http://%s/\n", WiFi.localIP().toString().c_str());
}

void handleWiFi()
{
  server.handleClient();
}

#endif
