/**
 * ESP32-S3 CAM - Main
 *
 * Camera capture with Cloud OCR via Vercel API
 *
 * Serial Commands (for testing):
 *   'o' or '1'-'4' - Run OCR and print numbers (1-4 selects index)
 *   'c' - Capture and show image info
 *   'w' - Show WiFi status
 *   'i' - Show I2C status
 *
 * I2C slave (0x42):
 *   - Receives: 2 bytes [current_slide (0-80), numpad_input (1-4)]
 *   - Returns: 1 byte (target_slide 0-80, 99 for retry, 0xFF for processing)
 */

#include <Arduino.h>
#include "camera_config.h"
#include "i2c_comm.h"
#include "vision/image.h"
#include "vision/transforms.h"
#include "vision/cloud_ocr.h"

#ifdef WIFI_DEBUG_MODE
#include "wifi_server.h"
#endif

// OCR Task handling
static TaskHandle_t ocrTaskHandle = NULL;
static volatile bool ocrInProgress = false;
static volatile uint8_t ocrNumpadInput = 0;
static volatile uint8_t ocrCurrentSlide = 0;
static volatile bool serialOcrRequest = false;

// OCR task - runs on core 0 (WiFi core)
void ocrTask(void *parameter)
{
    while (true)
    {
        // Wait for notification to start OCR
        ulTaskNotifyTake(pdTRUE, portMAX_DELAY);

        ocrInProgress = true;
        uint8_t numpad = ocrNumpadInput;
        uint8_t slide = ocrCurrentSlide;
        bool fromSerial = serialOcrRequest;
        serialOcrRequest = false;

        Serial.println("\n========== OCR START ==========");
        Serial.printf("Numpad: %d (index %d), Slide: %d\n", numpad, numpad - 1, slide);
        Serial.println("Calling Vercel API...");

        unsigned long startTime = millis();
        CloudOCRResult ocrResult = captureAndOCR();
        unsigned long elapsed = millis() - startTime;

        Serial.printf("OCR took %lu ms\n", elapsed);

        if (ocrResult.success && ocrResult.count > 0)
        {
            Serial.printf("\n>>> OCR SUCCESS: Found %d numbers <<<\n", ocrResult.count);
            Serial.print("Numbers: [");
            for (int i = 0; i < ocrResult.count; i++)
            {
                Serial.printf("%d%s", ocrResult.numbers[i], i < ocrResult.count - 1 ? ", " : "");
            }
            Serial.println("]");

            int index = numpad - 1;

            if (index >= 0 && index < ocrResult.count)
            {
                int value = ocrResult.numbers[index];
                uint8_t result = (uint8_t)min(max(value, 0), 80);
                Serial.printf("\nSelected: numbers[%d] = %d\n", index, result);

                if (!fromSerial)
                {
                    setI2CResult(result);
                    Serial.printf("I2C result set to: %d\n", result);
                }
            }
            else
            {
                Serial.printf("\n!!! Index %d out of range (only %d numbers) !!!\n", index, ocrResult.count);
                if (!fromSerial)
                {
                    setI2CResult(99);
                    Serial.println("I2C result set to: 99 (retry)");
                }
            }
        }
        else
        {
            Serial.printf("\n!!! OCR FAILED: %s !!!\n", ocrResult.error.c_str());
            if (!fromSerial)
            {
                setI2CResult(99);
                Serial.println("I2C result set to: 99 (retry)");
            }
        }

        Serial.println("========== OCR END ==========\n");
        ocrInProgress = false;
    }
}

void handleSerialCommand()
{
    if (Serial.available())
    {
        char cmd = Serial.read();

        switch (cmd)
        {
        case 'o':
        case 'O':
            // Run OCR with default index 1
            if (!ocrInProgress)
            {
                Serial.println("\n[Serial] Running OCR (index 0)...");
                ocrNumpadInput = 1;
                ocrCurrentSlide = 0;
                serialOcrRequest = true;
                xTaskNotifyGive(ocrTaskHandle);
            }
            else
            {
                Serial.println("[Serial] OCR already in progress!");
            }
            break;

        case '1':
        case '2':
        case '3':
        case '4':
            // Run OCR with specific index
            if (!ocrInProgress)
            {
                uint8_t idx = cmd - '0';
                Serial.printf("\n[Serial] Running OCR (index %d)...\n", idx - 1);
                ocrNumpadInput = idx;
                ocrCurrentSlide = 0;
                serialOcrRequest = true;
                xTaskNotifyGive(ocrTaskHandle);
            }
            else
            {
                Serial.println("[Serial] OCR already in progress!");
            }
            break;

        case 'c':
        case 'C':
            // Capture test
            {
                Serial.println("\n[Serial] Capturing image...");
                camera_fb_t *fb = esp_camera_fb_get();
                if (fb)
                {
                    Serial.printf("Captured: %dx%d, %d bytes, format=%d\n",
                                  fb->width, fb->height, fb->len, fb->format);
                    esp_camera_fb_return(fb);
                }
                else
                {
                    Serial.println("Capture failed!");
                }
            }
            break;

        case 'w':
        case 'W':
            // WiFi status
            Serial.println("\n[Serial] WiFi Status:");
            Serial.printf("  Connected: %s\n", WiFi.isConnected() ? "YES" : "NO");
            if (WiFi.isConnected())
            {
                Serial.printf("  SSID: %s\n", WiFi.SSID().c_str());
                Serial.printf("  IP: %s\n", WiFi.localIP().toString().c_str());
                Serial.printf("  RSSI: %d dBm\n", WiFi.RSSI());
            }
            break;

        case 'i':
        case 'I':
            // I2C status
            Serial.println("\n[Serial] I2C Status:");
            Serial.printf("  Address: 0x%02X\n", I2C_ADDR);
            Serial.printf("  SDA: GPIO%d, SCL: GPIO%d\n", I2C_SDA, I2C_SCL);
            Serial.printf("  Pending request: %s\n", hasI2CRequest() ? "YES" : "NO");
            Serial.printf("  Current result: %d\n", i2cResult);
            break;

        case 'h':
        case 'H':
        case '?':
            // Help
            Serial.println("\n=== Serial Commands ===");
            Serial.println("  o     - Run OCR (index 0)");
            Serial.println("  1-4   - Run OCR (index 0-3)");
            Serial.println("  c     - Capture test image");
            Serial.println("  w     - WiFi status");
            Serial.println("  i     - I2C status");
            Serial.println("  h/?   - This help");
            break;

        case '\n':
        case '\r':
            // Ignore newlines
            break;

        default:
            Serial.printf("[Serial] Unknown command: '%c' (send 'h' for help)\n", cmd);
            break;
        }
    }
}

void setup()
{
    Serial.begin(115200);
    delay(1000);
    Serial.println("\n=== ESP32-S3 CAM ===");
    Serial.println("Send 'h' for serial commands\n");

    // Camera
    if (!initCamera())
    {
        Serial.println("Camera FAILED");
        while (1)
            delay(1000);
    }
    Serial.println("Camera OK");

    // I2C slave
    initI2C();

#ifdef WIFI_DEBUG_MODE
    initWiFi();
#endif

    // Create OCR task on core 0 (same as WiFi)
    xTaskCreatePinnedToCore(
        ocrTask,
        "OCR",
        8192,
        NULL,
        1,
        &ocrTaskHandle,
        0);

    Serial.println("\nOCR task created on core 0");
    Serial.println("Ready! Send '1'-'4' to test OCR\n");
}

void loop()
{
    // Handle serial commands for testing
    handleSerialCommand();

    // Handle I2C requests
    if (hasI2CRequest())
    {
        ocrCurrentSlide = getI2CCurrentSlide();
        ocrNumpadInput = getI2CNumpadInput();
        clearI2CRequest();

        Serial.printf("[I2C] Request received: slide=%d, numpad=%d\n", ocrCurrentSlide, ocrNumpadInput);

        if (!ocrInProgress)
        {
            xTaskNotifyGive(ocrTaskHandle);
        }
        else
        {
            Serial.println("[I2C] OCR already in progress!");
        }
    }

#ifdef WIFI_DEBUG_MODE
    handleWiFi();
#endif

    delay(10);
}
